# Version 5.8 - Admin Promos
# Author: Code Omaha  
# Studio: Omaha Affiliates

import requests
import json
import sqlite3
import logging
import re
import os
import sys
import asyncio
import shutil
import glob
import random
import string
import telegram
from bs4 import BeautifulSoup
from datetime import datetime, timedelta, date
from urllib.parse import urlencode
from dotenv import load_dotenv
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import Application, CommandHandler, CallbackQueryHandler, MessageHandler, filters, ContextTypes
from support_tickets import *
from telegram.constants import ParseMode

def is_admin_only_mode():
    return os.path.exists("admin_only.lock")

def backup_database():
    now = datetime.now().strftime("%Y%m%d_%H%M%S")
    src = "bot_data.db"
    dst = f"bot_data_backup_{now}.db"
    try:
        shutil.copy(src, dst)
        print(f"🗄️ Database backup created: {dst}")
    except Exception as e:
        print(f"❌ Database backup failed: {e}")

def cleanup_old_backups(keep_latest=10):
    import glob
    files = sorted(glob.glob("bot_data_backup_*.db"), reverse=True)
    for f in files[keep_latest:]:
        try:
            os.remove(f)
            print(f"🧹 Removed old backup: {f}")
        except Exception as e:
            print(f"❌ Failed to remove backup {f}: {e}")

def check_db_sanity():
    conn = sqlite3.connect("bot_data.db")
    cur = conn.cursor()
    print("🔎 Running database sanity checks...")

    # 1. Check for duplicate approved links
    cur.execute('''
        SELECT panel_username, COUNT(*)
        FROM account_links
        WHERE status='approved'
        GROUP BY panel_username
        HAVING COUNT(*) > 1
    ''')
    dups = cur.fetchall()
    if dups:
        print("❌ WARNING: Duplicate approved links found!")
        for username, count in dups:
            print(f"  - {username} ({count} times)")
    else:
        print("✅ No duplicate approved links found.")

    # 2. Check that all necessary tables exist
    required_tables = ['users', 'account_links', 'orders', 'user_states']
    cur.execute("SELECT name FROM sqlite_master WHERE type='table';")
    tables = set(row[0] for row in cur.fetchall())
    for t in required_tables:
        if t not in tables:
            print(f"❌ WARNING: Missing required table: {t}")
    if all(t in tables for t in required_tables):
        print("✅ All important tables are present.")

    conn.close()

# Load environment variables
load_dotenv()

# Configure logging to reduce noise
logging.basicConfig(
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    level=logging.WARNING
)
logger = logging.getLogger(__name__)

async def handle_returning_user(update, context):  
    keyboard = [  
        [InlineKeyboardButton("🆕 New Customer", callback_data="new_customer")],  
        [InlineKeyboardButton("👤 User Login", callback_data="subscriber")]  
    ]  
    reply_markup = InlineKeyboardMarkup(keyboard)  
    bot_name = os.getenv("BOT_NAME", "Bot Management System")  
    await update.callback_query.edit_message_text(  
        f"👋 <b>Welcome to {bot_name}!</b>\n\n"  
        f"Use the buttons below to navigate.",  
        reply_markup=reply_markup,  
        parse_mode='HTML'  
    )

class PackageManager:
    def __init__(self, json_file: str = "packages.json"):
        self.json_file = json_file
        if not os.path.exists(self.json_file):
            print(f"❌ {self.json_file} not found! Please create the packages.json file.")
            return
        else:
            print(f"📦 Found existing packages.json file")
    
    def get_services(self):
        """Get active services from JSON file"""
        try:
            with open(self.json_file, 'r') as f:
                data = json.load(f)
                services = [service for service in data.get('services', []) if service.get('active', True)]
                return services
        except Exception as e:
            logger.error(f"Error reading packages.json: {e}")
            return []
    
    def get_service_by_id(self, service_id):
        """Get specific service by ID"""
        services = self.get_services()
        for service in services:
            if service['id'] == service_id:
                return service
        return None
    
    def get_service_by_name(self, service_name):
        """Get specific service by name"""
        services = self.get_services()
        for service in services:
            if service['name'] == service_name:
                return service
        return None
    
    def get_billing_cycle(self, service_id, cycle_id):
        """Get specific billing cycle for a service"""
        service = self.get_service_by_id(service_id)
        if service:
            for cycle in service.get('billing_cycles', []):
                if cycle['cycle'] == cycle_id:
                    return cycle
        return None
    
    def get_connection_option(self, service_id, cycle_id, connection_count):
        """Get specific connection option for matrix type services"""
        cycle = self.get_billing_cycle(service_id, cycle_id)
        if cycle and 'connections' in cycle:
            for conn in cycle['connections']:
                if conn['count'] == connection_count and conn.get('enabled', True):
                    return conn
        return None

import os
import logging

logger = logging.getLogger(__name__)

class PanelManager:
    def __init__(self, package_manager: 'PackageManager'):
        self.package_manager = package_manager
        self.panels = {}  # Cache for panel instances

    def get_panel_for_service(self, service_id: str):
        """Get the appropriate panel API instance for a service (Xtream or XUI)"""
        if service_id in self.panels:
            return self.panels[service_id]

        service = self.package_manager.get_service_by_id(service_id)
        if not service:
            logger.error(f"Service not found for ID: {service_id}")
            return None

        # Determine panel type (default to 'xtream' if not specified)
        panel_type = service.get('panel_type', 'xtream').lower()

        # Get panel credentials from service config or environment variables
        service_num = service_id.replace('service', '')
        panel_url = service.get('panel_url') or os.getenv(f"SERVICE{service_num}_PANEL_URL")
        panel_username = service.get('panel_username') or os.getenv(f"SERVICE{service_num}_PANEL_USERNAME")
        panel_password = service.get('panel_password') or os.getenv(f"SERVICE{service_num}_PANEL_PASSWORD")

        # For XUI, also get group_code and api_key
        group_code = service.get('group_code') or os.getenv(f"SERVICE{service_num}_GROUP_CODE")
        api_key = service.get('api_key') or os.getenv(f"SERVICE{service_num}_API_KEY")

        # Validate config - ensure all required values are present
        if panel_type == 'xui':
            if not all([panel_url, panel_username, panel_password, group_code, api_key]):
                logger.error(f"Missing XUI panel configuration for {service.get('name', service_id)} (service{service_num})")
                return None
            panel = XUIPanel(panel_url, panel_username, panel_password, group_code, api_key)
        else:  # Default to Xtream
            if not all([panel_url, panel_username, panel_password]):
                logger.error(f"Missing Xtream panel configuration for {service.get('name', service_id)} (service{service_num})")
                return None
            panel = XtreamPanelAPI(panel_url, panel_username, panel_password)

        self.panels[service_id] = panel
        logger.info(f"🔧 Panel configured for {service.get('name', service_id)}: {panel_url} (type: {panel_type})")
        return panel

class XtreamPanelAPI:
    def __init__(self, panel_url: str, username: str, password: str):
        self.panel_url = panel_url.rstrip('/')
        self.username = username
        self.password = password
        self.session = requests.Session()
        
    def authenticate(self):
        """Authenticate with the Xtream UI panel"""
        self.session = requests.Session()
        
        try:
            self.session.headers.update({
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:138.0) Gecko/20100101 Firefox/138.0',
                'Accept': 'application/json, text/javascript, */*; q=0.01',
                'Accept-Language': 'en-US,en;q=0.5',
                'X-Requested-With': 'XMLHttpRequest'
            })
            
            login_page = self.session.get(f"{self.panel_url}/login.php", timeout=10)
            
            login_data = {
                "username": self.username,
                "password": self.password,
                "remember_me": "on"
            }
            
            login_response = self.session.post(f"{self.panel_url}/login.php", data=login_data, timeout=10)
            
            if login_response.status_code == 200 and (len(login_response.history) > 0 or "dashboard" in login_response.text.lower()):
                return {"success": True, "message": "Authentication successful"}
            else:
                return {"success": False, "message": "Login failed"}
                
        except Exception as e:
            return {"success": False, "message": f"Authentication error: {str(e)}"}
    
    def search_user(self, username: str):
        """Search for a specific user using the DataTables endpoint"""
        auth_result = self.authenticate()
        if not auth_result["success"]:
            return {"error": f"Authentication failed: {auth_result['message']}"}
        
        try:
            params = {
                'draw': 1,
                'start': 0,
                'length': 250,
                'search[value]': username.lower(),
                'search[regex]': 'false',
                'id': 'users',
                'filter': '',
                'reseller': '',
                '_': str(int(datetime.now().timestamp() * 1000))
            }
            
            for i in range(12):
                params.update({
                    f'columns[{i}][data]': str(i),
                    f'columns[{i}][name]': '',
                    f'columns[{i}][searchable]': 'true',
                    f'columns[{i}][orderable]': 'true' if i != 11 else 'false',
                    f'columns[{i}][search][value]': '',
                    f'columns[{i}][search][regex]': 'false'
                })
            
            params.update({
                'order[0][column]': '0',
                'order[0][dir]': 'desc'
            })
            
            search_url = f"{self.panel_url}/table_search.php"
            response = self.session.get(search_url, params=params, timeout=15)
            
            if response.status_code != 200:
                return {"error": f"Search request failed with status {response.status_code}"}
            
            try:
                data = response.json()
            except json.JSONDecodeError:
                return {"error": "Invalid JSON response from panel"}
            
            if not isinstance(data, dict) or 'data' not in data:
                return {"error": "Unexpected response format"}
            
            for record in data['data']:
                if len(record) >= 8 and record[1].lower() == username.lower():
                    return {
                        "found": True,
                        "user_data": {
                            "id": record[0],
                            "username": record[1],
                            "password": record[2],
                            "reseller": record[3],
                            "active": record[4],
                            "enabled": record[5], 
                            "trial": record[6],
                            "expiration": record[7],
                            "connections": record[8] if len(record) > 8 else "0",
                            "max_connections": record[9] if len(record) > 9 else "1",
                            "last_login": record[10] if len(record) > 10 else "Never"
                        }
                    }
            
            
            return {"found": False, "message": f"No exact match for '{username}'"}
            
        except Exception as e:
            return {"error": f"Search error: {str(e)}"}
        
    def renew_user_account(self, username: str, package_id: str, duration_days: int = 30, notes: str = ""):
        """Renew existing account using the proper panel API"""
        
        # Authenticate first
        auth_result = self.authenticate()
        if not auth_result["success"]:
            return {"success": False, "error": f"Authentication failed: {auth_result['message']}"}
        
        try:
            logging.info(f"PANEL: Starting renewal for account {username}")
            logging.info(f"  Package ID: {package_id}")
            logging.info(f"  Duration: {duration_days} days")
            
            # Step 1: Find the user and get their ID
            search_result = self.search_user(username)
            if "error" in search_result:
                return {"success": False, "error": search_result["error"]}
            
            if not search_result.get("found"):
                return {"success": False, "error": f"User {username} not found in panel"}
            
            user_data = search_result["user_data"]
            user_id = user_data["id"]
            current_expiry = user_data.get("expiration", "")
            
            logging.info(f"PANEL: Found user {username} with ID {user_id}")
            logging.info(f"PANEL: Current expiry: {current_expiry}")
            
            # Step 2: Use the panel's API to get extension details
            api_url = f"{self.panel_url}/api.php"
            api_params = {
                'action': 'get_package',
                'package_id': package_id,
                'user_id': user_id
            }
            
            logging.info(f"PANEL: Getting package extension details from API")
            api_response = self.session.get(api_url, params=api_params, timeout=10)
            
            if api_response.status_code == 200:
                try:
                    api_data = api_response.json()
                    if api_data.get('result') is True:
                        package_data = api_data['data']
                        new_expiry = package_data.get('exp_date')
                        cost_credits = package_data.get('cost_credits', 0)
                        max_connections = package_data.get('max_connections', 1)
                        
                        logging.info(f"PANEL: API returned package extension data:")
                        logging.info(f"  New expiry: {new_expiry}")
                        logging.info(f"  Cost: {cost_credits} credits")
                        logging.info(f"  Max connections: {max_connections}")
                        
                        # Step 3: Submit the renewal form
                        edit_url = f"{self.panel_url}/user_reseller.php?id={user_id}"
                        
                        # Create form data for the renewal
                        renewal_form_data = {
                            'edit': user_id,
                            'username': user_data["username"],
                            'password': user_data["password"],
                            'member_id': '42',
                            'package': package_id,
                            'max_connections': str(max_connections),
                            'exp_date': new_expiry,
                            'reseller_notes': notes,
                            'submit_user': 'Purchase'
                        }
                        
                        logging.info(f"PANEL: Submitting renewal form")
                        
                        # Submit the renewal
                        renewal_response = self.session.post(
                            edit_url,
                            data=renewal_form_data,
                            timeout=15,
                            allow_redirects=True
                        )
                        
                        if renewal_response.status_code == 200:
                            logging.info(f"PANEL: Renewal submitted successfully")
                            
                            # Step 4: Verify the renewal worked
                            import time
                            time.sleep(3)  # Wait for processing
                            
                            verify_search = self.search_user(username)
                            if verify_search.get("found"):
                                updated_user_data = verify_search["user_data"]
                                updated_expiry = updated_user_data.get("expiration", "")
                                logging.info(f"PANEL: Verification - Updated expiry: {updated_expiry}")
                                
                                # Return success with details
                                return {
                                    "success": True,
                                    "username": username,
                                    "method": "api_renewal",
                                    "duration_added": duration_days,
                                    "package_id": package_id,
                                    "old_expiry": current_expiry,
                                    "new_expiry": updated_expiry if updated_expiry != current_expiry else new_expiry,
                                    "verified": True,
                                    "cost_credits": cost_credits,
                                    "message": "Account renewed successfully"
                                }
                            else:
                                return {"success": False, "error": "Account verification failed after renewal"}
                        else:
                            return {"success": False, "error": f"Renewal form submission failed: HTTP {renewal_response.status_code}"}
                    else:
                        return {"success": False, "error": f"API returned error: {api_data}"}
                except json.JSONDecodeError:
                    return {"success": False, "error": "API response is not valid JSON"}
            else:
                return {"success": False, "error": f"API call failed: HTTP {api_response.status_code}"}
                
        except Exception as e:
            logging.error(f"PANEL: Renewal error for {username}: {str(e)}")
            return {"success": False, "error": f"Renewal error: {str(e)}"}
        
    def create_user_account(self, username: str, password: str, package_id: str, duration_days: int = 30, max_connections: int = 1, notes: str = ""):
        """Create account using reseller purchase endpoint (auto-generates credentials)"""
        
        # Authenticate first
        auth_result = self.authenticate()
        if not auth_result["success"]:
            return {"success": False, "error": f"Authentication failed: {auth_result['message']}"}
        
        try:
            logging.info(f"PANEL: Creating account with package {package_id}")
            logging.info(f"  Duration: {duration_days} days")
            logging.info(f"  Max Connections: {max_connections}")
            
            # Step 1: Get package details via API (same as CAV)
            api_url = f"{self.panel_url}/api.php"
            api_params = {
                'action': 'get_package',
                'package_id': package_id
            }
            
            logging.info(f"PANEL: Getting package details from API")
            api_response = self.session.get(api_url, params=api_params, timeout=10)
            
            if api_response.status_code != 200:
                return {"success": False, "error": f"Could not get package details: HTTP {api_response.status_code}"}
                
            try:
                api_data = api_response.json()
                if api_data.get('result') is not True:
                    return {"success": False, "error": "Could not get package details from API"}
                    
                package_details = api_data['data']
                api_max_connections = package_details.get('max_connections', max_connections)
                api_exp_date = package_details.get('exp_date', '')
                
                logging.info(f"PANEL: API returned - Max Connections: {api_max_connections}, Exp Date: {api_exp_date}")
                
            except Exception as e:
                return {"success": False, "error": f"Error parsing package details: {str(e)}"}
            
            # Step 2: Submit form WITHOUT device fields (this is the key!)
            form_data = {
                'member_id': '42',  # Default member ID
                'package': str(package_id),
                'max_connections': str(api_max_connections),
                'exp_date': api_exp_date,
                'reseller_notes': notes,
                'submit_user': 'Purchase'
            }
            
            logging.info(f"PANEL: Submitting purchase form: {form_data}")
            
            # Submit to user_reseller.php endpoint
            response = self.session.post(
                f"{self.panel_url}/user_reseller.php",
                data=form_data,
                timeout=15,
                allow_redirects=True
            )
            
            logging.info(f"PANEL: Form submission response: {response.status_code}")
            logging.info(f"PANEL: Response URL: {response.url}")
            
            if response.status_code == 200:
                response_text = response.text
                
                # Check for success by looking for edit page redirect
                if "user_reseller.php?id=" in response.url:
                    # Extract user ID from URL
                    import re
                    user_id_match = re.search(r'user_reseller\.php\?id=(\d+)', response.url)
                    if user_id_match:
                        new_user_id = user_id_match.group(1)
                        logging.info(f"PANEL: Account created successfully! User ID: {new_user_id}")
                        
                        # Extract auto-generated credentials from the edit page
                        from bs4 import BeautifulSoup
                        soup = BeautifulSoup(response_text, 'html.parser')
                        
                        # Find username field
                        username_input = soup.find('input', {'name': 'username'})
                        auto_username = username_input.get('value', '') if username_input else ''
                        
                        # Find password field
                        password_input = soup.find('input', {'name': 'password'})
                        auto_password = password_input.get('value', '') if password_input else ''
                        
                        if auto_username and auto_password:
                            logging.info(f"PANEL: Auto-generated credentials: {auto_username} / {auto_password}")
                            
                            # Wait a moment then verify
                            import time
                            time.sleep(2)
                            
                            # Verify account exists
                            verify_result = self.search_user(auto_username)
                            if verify_result.get("found"):
                                return {
                                    "success": True,
                                    "username": auto_username,
                                    "password": auto_password,
                                    "user_id": new_user_id,
                                    "user_data": verify_result["user_data"],
                                    "message": "Account created successfully with auto-generated credentials"
                                }
                            else:
                                # Account creation succeeded but verification failed - still return success
                                return {
                                    "success": True,
                                    "username": auto_username,
                                    "password": auto_password,
                                    "user_id": new_user_id,
                                    "message": "Account created (verification pending)"
                                }
                        else:
                            return {"success": False, "error": "Could not extract auto-generated credentials"}
                    else:
                        return {"success": False, "error": "Could not extract user ID from redirect"}
                else:
                    # Check for errors in response
                    if "insufficient credits" in response_text.lower():
                        return {"success": False, "error": "Insufficient credits in panel"}
                    elif "error" in response_text.lower():
                        return {"success": False, "error": "Panel returned an error"}
                    else:
                        return {"success": False, "error": "No redirect to edit page - account may not have been created"}
            else:
                return {"success": False, "error": f"Form submission failed: HTTP {response.status_code}"}
                
        except Exception as e:
            logging.error(f"PANEL: Account creation exception: {str(e)}")
            return {"success": False, "error": f"Account creation error: {str(e)}"}
        
class PromoManager:  
    def __init__(self, filename="promos.json"):  
        self.filename = filename  
        self.promos = self.load_promos()  
  
    def load_promos(self):
        try:
            with open(self.filename, "r") as f:
                data = json.load(f)
                # Convert end_time strings to datetime objects
                for k, v in data.items():
                    v["end_time"] = datetime.fromisoformat(v["end_time"])
                return data
        except Exception:
            return {}
  
    def save_promos(self):  
        data = self.promos.copy()  
        for k, v in data.items():  
            v["end_time"] = v["end_time"].isoformat()  
        with open(self.filename, "w") as f:  
            json.dump(data, f, indent=2)  
  
    def set_promo(self, service_id, cycle_id, connection_count, sale_price, original_price, duration_minutes):  
        key = self._make_key(service_id, cycle_id, connection_count)  
        end_time = datetime.now() + timedelta(minutes=duration_minutes)  
        self.promos[key] = {  
            "sale_price": sale_price,  
            "original_price": original_price,  
            "end_time": end_time,  
        }  
        self.save_promos()  
  
    def clear_promo(self, service_id, cycle_id, connection_count):  
        key = self._make_key(service_id, cycle_id, connection_count)  
        if key in self.promos:  
            del self.promos[key]  
            self.save_promos()  
  
    def get_promo(self, service_id, cycle_id, connection_count):  
        key = self._make_key(service_id, cycle_id, connection_count)  
        promo = self.promos.get(key)  
        if promo and promo["end_time"] > datetime.now():  
            return promo  
        elif promo:  
            # Promo expired, clean up  
            self.clear_promo(service_id, cycle_id, connection_count)  
        return None  
  
    def _make_key(self, service_id, cycle_id, connection_count):  
        # Use None for connection_count if not applicable  
        return f"{service_id}|{cycle_id}|{connection_count if connection_count is not None else 'None'}"
    
promo_manager = PromoManager()

class Database:
    def __init__(self, db_path: str = "bot_data.db"):
        self.db_path = db_path
        self.init_db()
        self.migrate_db()
    
    def init_db(self):
        """Initialize database tables"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        # Users table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS users (
                user_id INTEGER PRIMARY KEY,
                username TEXT,
                telegram_username TEXT,
                is_admin BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')
        
        # Account linking table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS account_links (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                telegram_user_id INTEGER,
                panel_username TEXT,
                service_name TEXT DEFAULT 'Service1',
                status TEXT DEFAULT 'pending',
                verification_code TEXT,
                linked_at TIMESTAMP,
                approved_by INTEGER,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_checked TIMESTAMP,
                notes TEXT,
                UNIQUE(telegram_user_id, panel_username)
            )
        ''')
        
        # Orders table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS orders (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                telegram_user_id INTEGER,
                telegram_username TEXT,
                order_type TEXT,
                service_id TEXT,
                service_name TEXT,
                package_id TEXT,
                package_name TEXT,
                billing_cycle TEXT,
                panel_package_id TEXT,
                base_price REAL,
                xxx_addon BOOLEAN DEFAULT FALSE,
                setup_fee REAL DEFAULT 0,
                total_price REAL,
                username TEXT,
                password TEXT,
                details TEXT,
                status TEXT DEFAULT 'pending_payment',
                payment_status TEXT DEFAULT 'pending',
                payment_amount REAL,
                cashapp_tag TEXT,
                notes TEXT,
                fulfilled_at TIMESTAMP,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')
        
        # User states table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS user_states (
                user_id INTEGER PRIMARY KEY,
                state TEXT,
                data TEXT,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')

        cursor.execute("""
            CREATE TABLE IF NOT EXISTS renewal_notifications (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                telegram_user_id INTEGER,
                account_username TEXT,
                expiry_date TEXT,
                notified_on TEXT
            )
        """)
        
        # Wallet table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS wallet (
                user_id INTEGER PRIMARY KEY,
                balance REAL DEFAULT 0.0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')
        
        # Wallet transactions table
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS wallet_transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER,
                transaction_type TEXT,
                amount REAL,
                balance_before REAL,
                balance_after REAL,
                description TEXT,
                order_id INTEGER,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')

        cursor.execute('''
            CREATE TABLE IF NOT EXISTS auto_renewal_settings (
                user_id INTEGER PRIMARY KEY,
                enabled BOOLEAN DEFAULT FALSE,
                payment_timing TEXT DEFAULT 'manual',
                max_amount REAL DEFAULT 0.0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')

        cursor.execute('''
            CREATE TABLE IF NOT EXISTS auto_renewal_log (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER,
                account_username TEXT,
                service_name TEXT,
                status TEXT,
                amount REAL,
                wallet_balance_before REAL,
                wallet_balance_after REAL,
                error_message TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ''')

        # Subscriptions table for renewal tracking
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS subscriptions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER,
                username TEXT,
                service_id TEXT,
                service_name TEXT,
                package_id TEXT,
                cycle_id TEXT,
                connection_count INTEGER,
                purchase_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE(user_id, username, service_id)
            )
        ''')

        conn.commit()
        conn.close()
    
    # Global dictionary to track daily notifications
    daily_notifications = {}

    def has_been_notified_today(self, notification_key: str) -> bool:
        """Check if a notification has been sent today"""
        from datetime import date
        if notification_key not in self.daily_notifications:
            return False
        last_notified = self.daily_notifications[notification_key]
        return last_notified == date.today()

    def mark_notified_today(self, notification_key: str):
        """Mark that a notification has been sent today"""
        from datetime import date
        self.daily_notifications[notification_key] = date.today()

    def migrate_db(self):
        """Migrate existing database to add missing columns"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        # Check existing account_links table columns
        cursor.execute("PRAGMA table_info(account_links)")
        link_columns = [column[1] for column in cursor.fetchall()]
        
        missing_link_columns = [
            ('service_name', 'TEXT DEFAULT "Service1"'),
            ('last_checked', 'TIMESTAMP'),
            ('notes', 'TEXT'),
            ('expiry_date', 'TEXT')
        ]
        
        for column_name, column_type in missing_link_columns:
            if column_name not in link_columns:
                try:
                    cursor.execute(f"ALTER TABLE account_links ADD COLUMN {column_name} {column_type}")
                    print(f"✅ Added column '{column_name}' to account_links table")
                except sqlite3.OperationalError as e:
                    if "duplicate column name" not in str(e):
                        print(f"⚠️ Error adding column '{column_name}': {e}")
        
        # Existing orders table migration...
        cursor.execute("PRAGMA table_info(orders)")
        columns = [column[1] for column in cursor.fetchall()]
        
        missing_columns = [
            ('service_id', 'TEXT'),
            ('service_name', 'TEXT'),
            ('package_id', 'TEXT'),
            ('billing_cycle', 'TEXT'),
            ('panel_package_id', 'TEXT'),
            ('base_price', 'REAL'),
            ('xxx_addon', 'BOOLEAN DEFAULT FALSE'),
            ('setup_fee', 'REAL DEFAULT 0'),
            ('total_price', 'REAL'),
            ('password', 'TEXT'),
            ('payment_status', 'TEXT DEFAULT "pending"'),
            ('payment_amount', 'REAL'),
            ('cashapp_tag', 'TEXT'),
            ('fulfilled_at', 'TIMESTAMP'),
            ('telegram_username', 'TEXT')
        ]
        
        for column_name, column_type in missing_columns:
            if column_name not in columns:
                try:
                    cursor.execute(f"ALTER TABLE orders ADD COLUMN {column_name} {column_type}")
                    print(f"✅ Added column '{column_name}' to orders table")
                except sqlite3.OperationalError as e:
                    if "duplicate column name" not in str(e):
                        print(f"⚠️ Error adding column '{column_name}': {e}")
        
        conn.commit()
        conn.close()
        print("🔧 Database migration completed")

    # --- SUBSCRIPTION TRACKING METHODS ---
    def save_subscription(self, user_id: int, username: str, service_id: str, service_name: str,
                         package_id: str, cycle_id: str, connection_count: int = None):
        """Save or update a user's subscription details for renewal tracking."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('''
            INSERT OR REPLACE INTO subscriptions
            (user_id, username, service_id, service_name, package_id, cycle_id, connection_count, purchase_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
        ''', (user_id, username, service_id, service_name, package_id, cycle_id, connection_count))
        conn.commit()
        conn.close()

    def get_account_subscription(self, user_id: int, username: str):
        """Retrieve a user's subscription details for renewal."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('''
            SELECT package_id, cycle_id, connection_count
            FROM subscriptions
            WHERE user_id = ? AND username = ?
            ORDER BY purchase_date DESC
            LIMIT 1
        ''', (user_id, username))
        row = cursor.fetchone()
        conn.close()
        if row:
            return {
                "package_id": row[0],
                "cycle_id": row[1],
                "connection_count": row[2]
            }
        return {}

    def get_wallet_balance(self, user_id: int) -> float:
        """Get user's wallet balance"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("SELECT balance FROM wallet WHERE user_id = ?", (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result[0] if result else 0.0

    def add_wallet_funds(self, user_id: int, amount: float, description: str, order_id: int = None) -> bool:
        """Add funds to user's wallet"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            # Get current balance
            current_balance = self.get_wallet_balance(user_id)
            new_balance = current_balance + amount
            
            # Update or insert wallet balance
            cursor.execute("""
                INSERT OR REPLACE INTO wallet (user_id, balance, updated_at) 
                VALUES (?, ?, CURRENT_TIMESTAMP)
            """, (user_id, new_balance))
            
            # Record transaction
            cursor.execute("""
                INSERT INTO wallet_transactions 
                (user_id, transaction_type, amount, balance_before, balance_after, description, order_id)
                VALUES (?, 'credit', ?, ?, ?, ?, ?)
            """, (user_id, amount, current_balance, new_balance, description, order_id))
            
            conn.commit()
            return True
        except Exception as e:
            conn.rollback()
            print(f"Error adding wallet funds: {e}")
            return False
        finally:
            conn.close()

    def deduct_wallet_funds(self, user_id: int, amount: float, description: str, order_id: int = None) -> bool:
        """Deduct funds from user's wallet"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            # Ensure amount is a float
            amount = float(amount)
            
            # Get current balance
            cursor.execute("SELECT balance FROM wallet WHERE user_id = ?", (user_id,))
            result = cursor.fetchone()
            current_balance = float(result[0]) if result else 0.0
            
            # Check sufficient funds
            if current_balance < amount:
                return False
            
            new_balance = current_balance - amount
            
            # Update wallet balance
            cursor.execute("""
                INSERT OR REPLACE INTO wallet (user_id, balance, updated_at) 
                VALUES (?, ?, CURRENT_TIMESTAMP)
            """, (user_id, new_balance))
            
            # Record transaction
            cursor.execute("""
                INSERT INTO wallet_transactions 
                (user_id, transaction_type, amount, balance_before, balance_after, description, order_id)
                VALUES (?, 'debit', ?, ?, ?, ?, ?)
            """, (user_id, amount, current_balance, new_balance, description, order_id))
            
            conn.commit()
            return True
        except Exception as e:
            conn.rollback()
            logging.error(f"Error deducting wallet funds: {e}")
            return False
        finally:
            conn.close()

    def get_wallet_transactions(self, user_id: int, limit: int = 10):
        """Get recent wallet transactions for a user"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT transaction_type, amount, balance_after, description, created_at 
            FROM wallet_transactions 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        """, (user_id, limit))
        transactions = cursor.fetchall()
        conn.close()
        return transactions

    def add_user(self, user_id: int, telegram_username: str):
        """Add user to database while preserving admin status"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute("SELECT is_admin FROM users WHERE user_id = ?", (user_id,))
        existing_user = cursor.fetchone()
        
        if existing_user:
            cursor.execute(
                "UPDATE users SET telegram_username = ? WHERE user_id = ?",
                (telegram_username, user_id)
            )
        else:
            cursor.execute(
                "INSERT INTO users (user_id, telegram_username) VALUES (?, ?)",
                (user_id, telegram_username)
            )
        
        conn.commit()
        conn.close()
    
    def get_telegram_username(self, user_id: int) -> str:
        """Get telegram username for a user"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("SELECT telegram_username FROM users WHERE user_id = ?", (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result[0] if result and result[0] else "Unknown"

    def is_admin(self, user_id: int) -> bool:
        """Check if user is admin"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("SELECT user_id, is_admin FROM users WHERE user_id = ?", (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result and result[1]
    
    def set_admin(self, user_id: int, is_admin: bool = True):
        """Set user as admin"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "INSERT OR REPLACE INTO users (user_id, is_admin) VALUES (?, ?)",
            (user_id, is_admin)
        )
        conn.commit()
        conn.close()
    
    def get_linked_accounts(self, telegram_user_id: int):
        """
        Get all linked accounts for a Telegram user with service info,
        sorted first by service_name, then by expiry_date (soonest first, nulls last).
        """
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            """
            SELECT panel_username, service_name, expiry_date
            FROM account_links 
            WHERE telegram_user_id = ? AND status = 'approved'
            ORDER BY service_name ASC,
                CASE 
                    WHEN expiry_date IS NULL OR expiry_date = '' THEN 1
                    ELSE 0
                END,
                expiry_date ASC
            """,
            (telegram_user_id,)
        )
        accounts = cursor.fetchall()
        conn.close()
        # Return as (username, service_name), keep compatibility
        return [(row[0], row[1]) for row in accounts]

    def is_account_already_approved(self, panel_username, service_name=None):
        """Check if any approved account link exists for this panel account."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        if service_name:
            cursor.execute(
                "SELECT telegram_user_id FROM account_links WHERE panel_username = ? AND service_name = ? AND status = 'approved'",
                (panel_username, service_name)
            )
        else:
            cursor.execute(
                "SELECT telegram_user_id FROM account_links WHERE panel_username = ? AND status = 'approved'",
                (panel_username,)
            )
        result = cursor.fetchone()
        conn.close()
        return result is not None

    def get_account_service(self, telegram_user_id: int, panel_username: str):
        """Get the service name for a specific linked account"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "SELECT service_name FROM account_links WHERE telegram_user_id = ? AND panel_username = ? AND status = 'approved'",
            (telegram_user_id, panel_username)
        )
        result = cursor.fetchone()
        conn.close()
        return result[0] if result else os.getenv("SERVICE1_NAME", "Service1")  # Default to first service
    
    def can_access_account(self, telegram_user_id: int, panel_username: str) -> bool:
        """Check if a Telegram user can access a specific panel account"""
        if self.is_admin(telegram_user_id):
            return True
        
        linked_accounts = self.get_linked_accounts(telegram_user_id)
        for username, service_name in linked_accounts:
            if username == panel_username:
                return True
        return False
    
    def request_account_link(self, telegram_user_id: int, panel_username: str, verification_code: str, service_name: str = None):
        """Request to link a Telegram account to a panel account, with service specification"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        try:
            # Block if account is already approved for any user (considering service)
            if service_name:
                cursor.execute(
                    "SELECT telegram_user_id FROM account_links WHERE panel_username = ? AND service_name = ? AND status = 'approved'",
                    (panel_username, service_name)
                )
            else:
                cursor.execute(
                    "SELECT telegram_user_id FROM account_links WHERE panel_username = ? AND status = 'approved'",
                    (panel_username,)
                )
            existing = cursor.fetchone()
            if existing and existing[0] != telegram_user_id:
                conn.close()
                return "linked_to_other"

            # Check if this user has a pending/rejected request for this account
            if service_name:
                cursor.execute(
                    "SELECT id, status FROM account_links WHERE telegram_user_id = ? AND panel_username = ? AND service_name = ?",
                    (telegram_user_id, panel_username, service_name)
                )
            else:
                cursor.execute(
                    "SELECT id, status FROM account_links WHERE telegram_user_id = ? AND panel_username = ?",
                    (telegram_user_id, panel_username)
                )
            existing_request = cursor.fetchone()
            
            if existing_request:
                request_id, status = existing_request
                if status == 'pending':
                    conn.close()
                    return "already_pending"
                elif status == 'rejected':
                    # Update the rejected request to pending with new verification code
                    cursor.execute(
                        "UPDATE account_links SET status = 'pending', verification_code = ?, created_at = CURRENT_TIMESTAMP WHERE id = ?",
                        (verification_code, request_id)
                    )
                    conn.commit()
                    conn.close()
                    return "success"

            # Proceed to insert new request
            cursor.execute(
                "INSERT INTO account_links (telegram_user_id, panel_username, service_name, verification_code) VALUES (?, ?, ?, ?)",
                (telegram_user_id, panel_username, service_name or 'Service1', verification_code)
            )
            conn.commit()
            return "success"
        except sqlite3.IntegrityError:
            # Handle the unique constraint - update instead
            if service_name:
                cursor.execute(
                    "UPDATE account_links SET status = 'pending', verification_code = ?, created_at = CURRENT_TIMESTAMP WHERE telegram_user_id = ? AND panel_username = ? AND service_name = ?",
                    (verification_code, telegram_user_id, panel_username, service_name)
                )
            else:
                cursor.execute(
                    "UPDATE account_links SET status = 'pending', verification_code = ?, created_at = CURRENT_TIMESTAMP WHERE telegram_user_id = ? AND panel_username = ?",
                    (verification_code, telegram_user_id, panel_username)
                )
            conn.commit()
            return "success"
        finally:
            conn.close()
    
    def approve_account_link(self, link_id: int, admin_user_id: int):
        """Approve an account link request with security check."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "SELECT panel_username, service_name, telegram_user_id FROM account_links WHERE id = ?",
            (link_id,)
        )
        row = cursor.fetchone()
        if not row:
            conn.close()
            return False, "Request not found"
        panel_username, service_name, request_user_id = row

        cursor.execute(
            "SELECT id, telegram_user_id FROM account_links WHERE panel_username = ? AND service_name = ? AND status = 'approved'",
            (panel_username, service_name)
        )
        already = cursor.fetchone()
        if already:
            existing_link_id, linked_user_id = already
            conn.close()
            if linked_user_id == request_user_id:
                return False, "Account is already linked to this user."
            return False, f"This account is already linked to another user (user ID: {linked_user_id}). Unlink first."

        cursor.execute(
            "UPDATE account_links SET status = 'approved', approved_by = ?, linked_at = CURRENT_TIMESTAMP WHERE id = ?",
            (admin_user_id, link_id)
        )
        conn.commit()
        conn.close()
        return True, "Approved"

    def unlink_account(self, telegram_user_id: int, panel_username: str):
        """Unlink an account from a user"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE account_links SET status = 'unlinked' WHERE telegram_user_id = ? AND panel_username = ? AND status = 'approved'",
            (telegram_user_id, panel_username)
        )
        affected_rows = cursor.rowcount
        conn.commit()
        conn.close()
        return affected_rows > 0

    def remove_dead_account(self, telegram_user_id: int, panel_username: str):
        """Remove a dead account link (for accounts that don't exist in panel)"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE account_links SET status = 'removed_dead' WHERE telegram_user_id = ? AND panel_username = ? AND status = 'approved'",
            (telegram_user_id, panel_username)
        )
        affected_rows = cursor.rowcount
        conn.commit()
        conn.close()
        return affected_rows > 0

    def get_all_linked_accounts(self):
        """Get all linked accounts for pruning purposes (includes expiry if available)"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT telegram_user_id, panel_username, expiry_date, service_name, last_checked 
            FROM account_links 
            WHERE status = 'approved'
        """)
        accounts = cursor.fetchall()
        conn.close()
        return accounts

    def mark_account_checked(self, telegram_user_id: int, panel_username: str):
        """Mark account as recently checked"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE account_links SET last_checked = CURRENT_TIMESTAMP WHERE telegram_user_id = ? AND panel_username = ?",
            (telegram_user_id, panel_username)
        )
        conn.commit()
        conn.close()

    def prune_dead_account(self, telegram_user_id: int, panel_username: str, reason: str):
        """Automatically prune a dead account"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE account_links SET status = 'auto_pruned', notes = ? WHERE telegram_user_id = ? AND panel_username = ? AND status = 'approved'",
            (reason, telegram_user_id, panel_username)
        )
        affected_rows = cursor.rowcount
        conn.commit()
        conn.close()
        return affected_rows > 0
    
    def get_pending_link_requests(self):
        """Get all pending account link requests"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT al.id, al.telegram_user_id, al.panel_username, al.verification_code, al.created_at, u.telegram_username
            FROM account_links al
            LEFT JOIN users u ON al.telegram_user_id = u.user_id
            WHERE al.status = 'pending'
            ORDER BY al.created_at DESC
        """)
        requests = cursor.fetchall()
        conn.close()
        return requests
    
    def set_user_state(self, user_id: int, state: str, data: str = ""):
        """Set user state"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "INSERT OR REPLACE INTO user_states (user_id, state, data, updated_at) VALUES (?, ?, ?, CURRENT_TIMESTAMP)",
            (user_id, state, data)
        )
        conn.commit()
        conn.close()
    
    def get_user_state(self, user_id: int):
        """Get user state"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("SELECT state, data FROM user_states WHERE user_id = ?", (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result if result else (None, None)
    
    def clear_user_state(self, user_id: int):
        """Clear user state"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM user_states WHERE user_id = ?", (user_id,))
        conn.commit()
        conn.close()
    
    def add_order_with_payment(self, telegram_user_id: int, telegram_username: str, order_type: str, service_id: str, service_name: str, 
                              package_id: str, package_name: str, billing_cycle: str, panel_package_id: str, 
                              base_price: float, xxx_addon: bool, setup_fee: float, total_price: float, 
                              cashapp_tag: str, username: str = "", password: str = "", details: str = ""):
        """Add new order with payment tracking and random Bot payment reference"""
        import random
        
        # Generate random 4-digit Bot payment reference
        bot_payment_ref = f"Bot{random.randint(1000, 9999)}"
        
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            """INSERT INTO orders (telegram_user_id, telegram_username, order_type, service_id, service_name, package_id, package_name, 
               billing_cycle, panel_package_id, base_price, xxx_addon, setup_fee, total_price, payment_amount, 
               cashapp_tag, username, password, details, status, payment_status, notes) 
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending_payment', 'pending', ?)""",
            (telegram_user_id, telegram_username, order_type, service_id, service_name, package_id, package_name, billing_cycle, 
             panel_package_id, base_price, xxx_addon, setup_fee, total_price, total_price, cashapp_tag, 
             username, password, details, bot_payment_ref)  # Store Bot#### in notes field
        )
        order_id = cursor.lastrowid
        conn.commit()
        conn.close()
        return order_id, bot_payment_ref  # Return both order ID and payment reference
    
    def get_order_by_id(self, order_id: int):
        """Get order by ID and return as dictionary"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT id, telegram_user_id, telegram_username, order_type, service_id, 
                service_name, package_id, package_name, billing_cycle, panel_package_id,
                base_price, xxx_addon, setup_fee, total_price, username, password,
                details, status, payment_status, payment_amount, cashapp_tag, notes,
                fulfilled_at, created_at, updated_at
            FROM orders 
            WHERE id = ?
        """, (order_id,))
        order = cursor.fetchone()
        conn.close()
        
        if not order:
            return None
        
        # Return as dictionary for easier access
        return {
            'id': order[0],
            'telegram_user_id': order[1],
            'telegram_username': order[2],
            'order_type': order[3],
            'service_id': order[4],
            'service_name': order[5],
            'package_id': order[6],
            'package_name': order[7],
            'billing_cycle': order[8],
            'panel_package_id': order[9],
            'base_price': order[10],
            'xxx_addon': order[11],
            'setup_fee': order[12],
            'total_price': order[13],
            'username': order[14],
            'password': order[15],
            'details': order[16],
            'status': order[17],
            'payment_status': order[18],
            'payment_amount': order[19],
            'cashapp_tag': order[20],
            'notes': order[21],
            'fulfilled_at': order[22],
            'created_at': order[23],
            'updated_at': order[24]
        }
    
    def get_user_orders(self, telegram_user_id: int):
        """Get all orders for a specific user"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT id, order_type, service_name, package_name, billing_cycle, total_price, status, 
                   payment_status, created_at, xxx_addon, username, password
            FROM orders 
            WHERE telegram_user_id = ? 
            ORDER BY created_at DESC
        """, (telegram_user_id,))
        orders = cursor.fetchall()
        conn.close()
        return orders
    
    def get_pending_orders(self):
        """Get all pending orders"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT o.id, o.telegram_user_id, o.order_type, o.service_name, o.package_name, o.billing_cycle, o.total_price, o.username, o.details, o.created_at, u.telegram_username 
            FROM orders o 
            LEFT JOIN users u ON o.telegram_user_id = u.user_id 
            WHERE o.status = 'pending_payment' OR o.status = 'pending'
            ORDER BY o.created_at DESC
        """)
        orders = cursor.fetchall()
        conn.close()
        return orders
    
    def update_order_status(self, order_id: int, status: str, notes: str = ""):
        """Update order status"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE orders SET status = ?, notes = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
            (status, notes, order_id)
        )
        conn.commit()
        conn.close()

    def get_auto_renewal_settings(self, user_id: int):
        """Get user's auto-renewal settings"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT enabled, payment_timing, max_amount 
            FROM auto_renewal_settings 
            WHERE user_id = ?
        """, (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result

    def set_auto_renewal_setting(self, user_id: int, enabled: bool, timing: str):
        """Set user's auto-renewal preference"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            INSERT OR REPLACE INTO auto_renewal_settings 
            (user_id, enabled, payment_timing, updated_at) 
            VALUES (?, ?, ?, CURRENT_TIMESTAMP)
        """, (user_id, enabled, timing))
        conn.commit()
        conn.close()

    def log_auto_renewal_attempt(self, user_id: int, username: str, service: str, 
                                status: str, amount: float, balance_before: float, 
                                balance_after: float, error: str = None):
        """Log auto-renewal attempt"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO auto_renewal_log 
            (user_id, account_username, service_name, status, amount, 
            wallet_balance_before, wallet_balance_after, error_message)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        """, (user_id, username, service, status, amount, balance_before, balance_after, error))
        conn.commit()
        conn.close()

# Configuration from environment variables
BOT_TOKEN = os.getenv("BOT_TOKEN")
BOT_NAME = os.getenv("BOT_NAME", "Bot Management System")

# Parse admin user IDs from comma-separated string
admin_ids_str = os.getenv("ADMIN_USER_IDS", "")
ADMIN_USER_IDS = [int(id.strip()) for id in admin_ids_str.split(",") if id.strip().isdigit()]

if not BOT_TOKEN:
    raise ValueError("BOT_TOKEN environment variable is required")
if not ADMIN_USER_IDS:
    raise ValueError("ADMIN_USER_IDS environment variable is required")

# Initialize
package_manager = PackageManager()
panel_manager = PanelManager(package_manager)
db = Database()

# FORCE SET ADMIN RIGHTS
print("🔧 Force setting admin rights...")
for admin_id in ADMIN_USER_IDS:
    db.set_admin(admin_id, True)
    print(f"✅ Admin rights forcefully set for user {admin_id}")

def parse_order_fields(order):
    """Parse order tuple into a dictionary for easier access"""
    if not order:
        return None
    
    # Map indices to field names based on actual database schema
    return {
        'id': order[0],
        'telegram_user_id': order[1],
        'telegram_username': order[2],
        'order_type': order[3],
        'service_id': order[4],
        'service_name': order[5],
        'package_id': order[6],
        'package_name': order[7],
        'billing_cycle': order[8],
        'panel_package_id': order[9],
        'base_price': order[10],
        'xxx_addon': order[11],
        'setup_fee': order[12],
        'total_price': order[13],
        'username': order[14],
        'password': order[15],
        'details': order[16],
        'status': order[17],
        'payment_status': order[18],
        'payment_amount': order[19],
        'cashapp_tag': order[20],
        'notes': order[21],
        'fulfilled_at': order[22],
        'created_at': order[23],
        'updated_at': order[24]
    }

def generate_username_password():
    """Generate random username and password for new accounts"""
    import random
    import string
    
    username = ''.join(random.choices(string.ascii_lowercase + string.digits, k=8))
    password = ''.join(random.choices(string.ascii_letters + string.digits, k=12))
    return username, password

def clean_html_content(content):
    """Clean HTML content for safe display in Telegram"""
    if not content:
        return "Unknown"
    
    content = str(content)
    import html
    content = html.unescape(content)
    content = re.sub(r'<[^>]+>', '', content)
    content = re.sub(r'\s+', ' ', content).strip()
    
    return content if content else "Unknown"

def clean_status(status_html):
    """Clean HTML status indicators to simple text"""
    if not status_html:
        return "Unknown"
    
    status_clean = clean_html_content(status_html).lower()
    
    if 'success' in status_clean or 'active' in status_clean:
        return "Active"
    elif 'warning' in status_clean or 'expired' in status_clean:
        return "Warning"
    elif 'secondary' in status_clean or 'inactive' in status_clean:
        return "Inactive"
    else:
        return "Unknown"

def parse_expiry_date(expiry_str):
    """Parse expiry date string to datetime object"""
    try:
        if not expiry_str or expiry_str in ["Never", "0000-00-00", "0000-00-00 00:00:00", "", "null", "NULL"]:
            return None
        
        # Try multiple date formats that different panels might use
        date_formats = [
            "%Y-%m-%d %H:%M:%S",  # Standard: 2024-01-15 23:59:59
            "%Y-%m-%d %H:%M",     # Format without seconds: 2025-07-27 23:18
            "%Y-%m-%d",           # Date only: 2024-01-15
            "%m/%d/%Y",           # US format: 01/15/2024
            "%d/%m/%Y",           # EU format: 15/01/2024
            "%Y%m%d",             # Compact: 20240115
        ]
        
        for date_format in date_formats:
            try:
                parsed_date = datetime.strptime(expiry_str, date_format)
                print(f"SUCCESS: Parsed '{expiry_str}' using format '{date_format}' -> {parsed_date}")
                return parsed_date
            except ValueError:
                continue
        
        # If no format worked, log it and return None
        print(f"WARNING: Could not parse expiry date '{expiry_str}' with any known format - renewal will be blocked")
        return None
        
    except Exception as e:
        print(f"ERROR: Exception parsing expiry date '{expiry_str}': {e}")
        return None

def is_within_renewal_window(expiry_date, days=10):
    """Check if account is within renewal window (expires within X days)"""
    if not expiry_date:
        print(f"WARNING: No expiry date provided - blocking renewal (strict policy)")
        return False
    
    now = datetime.now()
    time_until_expiry = expiry_date - now
    days_left = time_until_expiry.days
    
    print(f"DEBUG: Expiry check - Date: {expiry_date}, Days left: {days_left}, Within window ({days} days): {days_left <= days}")
    
    # Allow renewal if within the specified days OR already expired
    return days_left <= days

def check_renewal_eligibility(username, service_name, panel_manager):
    """Centralized renewal eligibility check with detailed logging"""
    
    # First, try to get service directly from package manager by name
    service = package_manager.get_service_by_name(service_name)
    
    if not service:
        # Try to map legacy names to current service names
        legacy_mapping = {}
        for service_num in range(1, 6):
            env_service_name = os.getenv(f"SERVICE{service_num}_NAME")
            if env_service_name:
                # Map both the env name and common legacy names to service ID
                legacy_mapping[env_service_name] = f"service{service_num}"
        
        service_id = legacy_mapping.get(service_name)
        if service_id:
            service = package_manager.get_service_by_id(service_id)
    
    if not service:
        return False, f"Unknown service: {service_name}"
    
    service_id = service['id']
    panel = panel_manager.get_panel_for_service(service_id)
    
    if not panel:
        return False, f"No panel configured for {service_name}"
    
    result = panel.search_user(username)
    
    if "error" in result:
        return False, f"Panel error: {result['error']}"
    
    if not result.get('found'):
        return False, "Account not found in panel"
    
    user_data = result['user_data']
    expiry_str = user_data.get('expiration', '')
    
    print(f"RENEWAL CHECK: {service_name} account '{username}' expiry raw: '{expiry_str}'")
    
    # Parse expiry date
    expiry_date = parse_expiry_date(expiry_str)
    
    if expiry_date is None:
        return False, f"No valid expiration date found - contact support"
    
    # Check if within 10-day renewal window
    if is_within_renewal_window(expiry_date, 10):
        now = datetime.now()
        days_left = (expiry_date - now).days
        
        if days_left > 0:
            return True, f"Renewal allowed (expires in {days_left} days)"
        else:
            days_expired = abs(days_left)
            return True, f"Renewal allowed (expired {days_expired} days ago)"
    else:
        now = datetime.now()
        days_left = (expiry_date - now).days
        return False, f"Renewal available in {days_left - 10} days (expires in {days_left} days)"

def is_expired_for_days(expiry_date, days=14):
    """Check if account has been expired for more than X days"""
    if not expiry_date:
        return False  # Never expires
    
    now = datetime.now()
    if expiry_date > now:
        return False  # Not expired yet
    
    days_expired = (now - expiry_date).days
    return days_expired > days

def get_main_menu_keyboard(user_id):
    """Get main menu keyboard with wallet option"""
    # Get wallet balance
    balance = db.get_wallet_balance(user_id)
    
    keyboard = [
        # Row 1: My Accounts | Dashboard
        [
            InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts"),
            InlineKeyboardButton("📊 Dashboard", callback_data="dashboard_menu")
        ],
        # Row 2: Wallet | Renew Account  
        [
            InlineKeyboardButton(f"💰 Wallet (${balance:.2f})", callback_data="wallet"),
            InlineKeyboardButton("🔄 Renew Account", callback_data="renew_account")
        ],
        # Row 3: New | My Orders
        [
            InlineKeyboardButton("🛒 New Order", callback_data="new_order"),
            InlineKeyboardButton("📋 My Orders", callback_data="my_orders")
        ],
        # Row 4: Offers | Support Ticket
        [
            InlineKeyboardButton("🔥 Offers", callback_data="offers"),
            InlineKeyboardButton("🆘 Support Tickets", callback_data="support_tickets")
        ]
    ]
    
    # Row 5: Admin Panel (if admin) - full width
    if db.is_admin(user_id):
        keyboard.append([InlineKeyboardButton("👨‍💼 Admin Panel", callback_data="admin_panel")])
    
    # Row 6: Help, Tools and About
    keyboard.append([InlineKeyboardButton("ℹ️ Help & Support", callback_data="help")])
    keyboard.append([InlineKeyboardButton("ℹ️ About", callback_data="about_page")])
    keyboard.append([InlineKeyboardButton("« Back to Start", callback_data="back_to_start")])
    
    return InlineKeyboardMarkup(keyboard)

def get_back_button_keyboard(back_to="menu"):
    """Get consistent back button for sub-menus"""
    if back_to == "start":
        return InlineKeyboardMarkup([[InlineKeyboardButton("« Back to Start", callback_data="back_to_start")]])
    elif back_to == "menu":
        return InlineKeyboardMarkup([[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]])
    else:
        return InlineKeyboardMarkup([[InlineKeyboardButton(f"« Back to {back_to}", callback_data=f"back_to_{back_to}")]])

def get_initial_keyboard():
    """Get initial selection keyboard"""
    keyboard = [
        [InlineKeyboardButton("🆕 New Customer", callback_data="new_customer")],
        [InlineKeyboardButton("👤 User Login", callback_data="subscriber")]     
    ]
    return InlineKeyboardMarkup(keyboard)

async def prune_dead_accounts():
    """Automatically prune accounts that have been expired for 14+ days"""
    print("🧹 Starting automatic account pruning...")
    
    pruned_count = 0
    checked_count = 0
    
    # Get all linked accounts
    all_accounts = db.get_all_linked_accounts()
    
    for telegram_user_id, panel_username, expiry_date, service_name, last_checked in all_accounts:
        checked_count += 1
        
        # Skip if checked recently (within last 24 hours) unless it's been a week
        if last_checked:
            last_check_time = datetime.fromisoformat(last_checked)
            hours_since_check = (datetime.now() - last_check_time).total_seconds() / 3600
            if hours_since_check < 24:
                continue
        
        # Get service panel
        service = package_manager.get_service_by_name(service_name)
        if not service:
            continue
            
        service_id = service['id']
        panel = panel_manager.get_panel_for_service(service_id)
        
        if not panel:
            continue
        
        # Search for account in panel
        result = panel.search_user(panel_username)
        
        # Mark as checked
        db.mark_account_checked(telegram_user_id, panel_username)
        
        if "error" in result:
            print(f"⚠️ Error checking account {panel_username}: {result['error']}")
            continue
        
        should_prune = False
        prune_reason = ""
        
        if not result.get('found'):
            # Account not found in panel
            should_prune = True
            prune_reason = "Account not found in panel"
        else:
            # Account found - check if expired for 14+ days
            user_data = result['user_data']
            expiry_date = parse_expiry_date(user_data.get('expiration'))
            
            if is_expired_for_days(expiry_date, 14):
                should_prune = True
                prune_reason = f"Account expired for 14+ days (expired: {user_data.get('expiration')})"
        
        if should_prune:
            if db.prune_dead_account(telegram_user_id, panel_username, prune_reason):
                pruned_count += 1
                print(f"🗑️ Auto-pruned: {panel_username} ({service_name}) from user {telegram_user_id} - {prune_reason}")
    
    if pruned_count > 0:
        print(f"🧹 Pruning complete: {pruned_count} dead accounts removed (checked {checked_count} accounts)")
    else:
        print(f"🧹 Pruning complete: No accounts needed removal (checked {checked_count} accounts)")

async def dashboard_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    # Choose user object (works for both command and callback)
    user = (update.effective_user if update.effective_user else
            (update.message.from_user if hasattr(update, "message") and update.message else None))
    user_id = user.id if user else None

    # --- Show loading message immediately ---
    loading_msg = None
    if hasattr(update, "callback_query") and update.callback_query:
        loading_msg = await update.callback_query.edit_message_text(
            "⏳ <b>Loading your dashboard...</b>",
            parse_mode="HTML"
        )
    elif hasattr(update, "message") and update.message:
        loading_msg = await update.message.reply_text(
            "⏳ <b>Loading your dashboard...</b>",
            parse_mode="HTML"
        )

    # Wallet Balance
    balance = db.get_wallet_balance(user_id)

    # Linked Accounts with Expiry Info
    linked_accounts = db.get_linked_accounts(user_id)
    accounts_info = ""
    renewal_eligible = []
    for username, service_name in linked_accounts:
        # Fetch expiry date from panel (if available)
        service = package_manager.get_service_by_name(service_name)
        expiry_str = None
        days_left = None
        expiry_hint = ""
        if service:
            service_id = service['id']
            panel = panel_manager.get_panel_for_service(service_id)
            if panel:
                result = panel.search_user(username)
                if result.get('found'):
                    expiry_str = result['user_data'].get('expiration')
                    # Parse days left
                    from datetime import datetime
                    def parse_expiry_date(expiry_str):
                        try:
                            if not expiry_str or expiry_str in ["Never", "0000-00-00", "0000-00-00 00:00:00", "", "null", "NULL"]:
                                return None
                            date_formats = [
                                "%Y-%m-%d %H:%M:%S",
                                "%Y-%m-%d %H:%M",
                                "%Y-%m-%d",
                                "%m/%d/%Y",
                                "%d/%m/%Y",
                                "%Y%m%d",
                            ]
                            for fmt in date_formats:
                                try:
                                    return datetime.strptime(expiry_str, fmt)
                                except ValueError:
                                    continue
                        except Exception:
                            return None
                    expiry_dt = parse_expiry_date(expiry_str)
                    if expiry_dt:
                        days_left = (expiry_dt - datetime.now()).days
                        if days_left < 0:
                            expiry_hint = f"(expired {abs(days_left)}d ago)"
                        else:
                            expiry_hint = f"(expires in {days_left}d)"
                        # Renewal eligibility
                        if 0 <= days_left <= 10:
                            renewal_eligible.append((username, service_name))
                else:
                    expiry_hint = "(not found)"
            else:
                expiry_hint = ""
        accounts_info += f"• <b>{service_name}</b>: <code>{username}</code> {expiry_hint}\n"

    if not accounts_info:
        accounts_info = "No linked accounts."

    # Recent Orders
    orders = db.get_user_orders(user_id)
    orders_info = ""
    for order in orders[:3]:
        order_id, order_type, service_name, package_name, billing_cycle, total_price, status, payment_status, created_at, xxx_addon, username, password = order
        try:
            order_date = datetime.fromisoformat(created_at).strftime("%m/%d")
        except:
            order_date = created_at[:10]
        status_emoji = "✅" if status == "completed" else ("❌" if status == "cancelled" else "⏳")
        orders_info += f"{status_emoji} #{order_id} {service_name} {package_name} (${total_price:.2f}) {order_date}\n"
    if not orders_info:
        orders_info = "No recent orders."

    # Dashboard Message
    dashboard = f"""👤 <b>Welcome, {user.first_name}!</b>

💰 <b>Wallet:</b> <code>${balance:.2f}</code>

📱 <b>Accounts:</b>
{accounts_info}

📋 <b>Recent Orders:</b>
{orders_info}
"""

    # Inline Buttons
    keyboard = [
        [InlineKeyboardButton("💰 Add Funds", callback_data="add_funds")],
        [InlineKeyboardButton("🛒 New Order", callback_data="new_order")],
        [InlineKeyboardButton("📋 My Orders", callback_data="my_orders")],
    ]
    if renewal_eligible:
        for username, service_name in renewal_eligible:
            keyboard.append([InlineKeyboardButton(f"🔄 Renew {username} ({service_name})", callback_data=f"renew_user_{username}")])
    keyboard.append([InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")])

    reply_markup = InlineKeyboardMarkup(keyboard)

    # --- Always edit the loading message to the real dashboard ---
    if loading_msg:
        await loading_msg.edit_text(
            dashboard,
            reply_markup=reply_markup,
            parse_mode="HTML"
        )

async def process_direct_renewal(context, chat_id, message_id, user_id, username):
    """Process direct renewal with service-specific settings and renewal window validation"""
    try:
        # Get the service for this account
        service_name = db.get_account_service(user_id, username)
        service = package_manager.get_service_by_name(service_name)
        
        if not service:
            keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text="❌ <b>Renewal Error</b>\n\nCould not determine account service. Please contact support.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return
        
        print(f"🔍 Processing renewal for {username} - Service: {service_name}")
        
        # Check if account exists and get expiry date
        service_id = service['id']
        panel = panel_manager.get_panel_for_service(service_id)
        
        if not panel:
            keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text="❌ <b>Renewal Error</b>\n\nCould not connect to panel. Please contact support.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return
        
        # Search for account in panel
        result = panel.search_user(username)
        if "error" in result:
            keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text=f"❌ <b>Renewal Error</b>\n\nError checking account: {result['error']}",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return
        
        if not result.get('found'):
            keyboard = [
                [InlineKeyboardButton("🗑️ Remove Dead Account", callback_data=f"remove_dead_{username}")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text=f"❌ <b>Account Not Found</b>\n\nAccount '{username}' was not found in the {service_name} panel.\n\nThis account may have been deleted or expired. You can remove it from your linked accounts.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return
        
        # Check renewal eligibility (within 10 days of expiry)
        user_data = result['user_data']
        expiry_date = parse_expiry_date(user_data.get('expiration'))
        if not is_within_renewal_window(expiry_date, 10):
            days_until_expiry = (expiry_date - datetime.now()).days if expiry_date else "unlimited"
            keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text=f"⏰ <b>Renewal Not Available</b>\n\n"
                     f"Account: {username}\n"
                     f"Expires: {user_data.get('expiration', 'Never')}\n"
                     f"Days until expiry: {days_until_expiry}\n\n"
                     f"🔒 <b>Renewals are only allowed within 10 days of expiry.</b>\n\n"
                     f"Please try again when your account is closer to expiration.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return

        # --- NEW: Get user's original subscription info ---
        # You must have stored these when the user first ordered!
        # Example: {'cycle_id': '90days', 'connection_count': 2}
        subscription_info = db.get_account_subscription(user_id, username)
        user_cycle_id = subscription_info.get('cycle_id')
        user_connection_count = subscription_info.get('connection_count')  # May be None for non-matrix

        # Use service-specific settings based on order type
        if service.get('order_type') == 'direct_cycles':
            cycles = service.get('billing_cycles', [])
            if not cycles:
                keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await context.bot.edit_message_text(
                    chat_id=chat_id,
                    message_id=message_id,
                    text="❌ <b>Renewal Error</b>\n\nNo billing cycles available for this service.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return

            # Find the cycle the user originally purchased
            cycle = next((c for c in cycles if c['cycle'] == user_cycle_id), None)
            if not cycle:
                keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await context.bot.edit_message_text(
                    chat_id=chat_id,
                    message_id=message_id,
                    text="❌ <b>Renewal Error</b>\n\nYour original billing cycle is no longer available. Please contact support.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return

            cycle_id = cycle['cycle']
            panel_package_id = cycle.get('panel_package_id', '')

            package = {
                'id': service['id'],
                'name': service['name'],
                'description': f"Renewal for {service['name']}"
            }
            package_id = service['id']

            base_price = cycle['price']

        else:
            # For matrix services, use packages
            packages = service.get('packages', [])
            if not packages:
                keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await context.bot.edit_message_text(
                    chat_id=chat_id,
                    message_id=message_id,
                    text="❌ <b>Renewal Error</b>\n\nNo packages available for this service.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return

            # Find the package the user originally purchased (if you store package_id)
            package_id = subscription_info.get('package_id', packages[0]['id'])
            package = next((p for p in packages if p['id'] == package_id), packages[0])
            cycles = package.get('billing_cycles', []) or service.get('billing_cycles', [])
            if not cycles:
                keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await context.bot.edit_message_text(
                    chat_id=chat_id,
                    message_id=message_id,
                    text="❌ <b>Renewal Error</b>\n\nNo billing cycles available for this service.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return

            # Find the cycle the user originally purchased
            cycle = next((c for c in cycles if c['cycle'] == user_cycle_id), None)
            if not cycle:
                keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await context.bot.edit_message_text(
                    chat_id=chat_id,
                    message_id=message_id,
                    text="❌ <b>Renewal Error</b>\n\nYour original billing cycle is no longer available. Please contact support.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return

            cycle_id = cycle['cycle']

            # For matrix: find the connection the user originally purchased
            if 'connections' in cycle and user_connection_count is not None:
                connection = next((conn for conn in cycle['connections'] if conn['count'] == user_connection_count), None)
                if not connection:
                    keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
                    reply_markup = InlineKeyboardMarkup(keyboard)
                    await context.bot.edit_message_text(
                        chat_id=chat_id,
                        message_id=message_id,
                        text="❌ <b>Renewal Error</b>\n\nYour original connection option is no longer available. Please contact support.",
                        reply_markup=reply_markup,
                        parse_mode='HTML'
                    )
                    return
                panel_package_id = connection.get('panel_package_no_xxx', '')
                base_price = connection['base_price']
            else:
                panel_package_id = cycle.get('panel_package_id', '')
                base_price = cycle['price']

        setup_fee = 0  # No setup fee for renewals
        total_price = base_price + setup_fee
        xxx_addon = False  # Default to no XXX

        # Dynamic service emoji based on service number
        service_emoji = "📱"  # Default
        for service_num in range(1, 6):
            if service['id'] == f"service{service_num}":
                emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                service_emoji = emoji_map.get(service_num, "📱")
                break

        confirmation_text = f"🔄 <b>Account Renewal Confirmation</b>\n\n"
        confirmation_text += f"<b>Renewing Account:</b> {username}\n"
        confirmation_text += f"{service_emoji} <b>Service:</b> {service['name']}\n"
        confirmation_text += f"<b>Current Expiry:</b> {user_data.get('expiration', 'Unknown')}\n"
        confirmation_text += f"<b>Package:</b> {package['name']}\n"
        confirmation_text += f"<b>Description:</b> {package['description']}\n"
        confirmation_text += f"<b>Billing Cycle:</b> {cycle['name']}\n"

        if service.get('xxx_support', False):
            confirmation_text += f"<b>XXX Content:</b> {'Yes' if xxx_addon else 'No'}\n\n"
        else:
            confirmation_text += "\n"

        confirmation_text += f"💰 <b>Renewal Pricing:</b>\n"
        confirmation_text += f"• Package Price: ${base_price:.2f}\n"
        confirmation_text += f"• Setup Fee: $0.00 (waived for renewals)\n"
        confirmation_text += f"<b>Total: ${total_price:.2f}</b>\n\n"
        confirmation_text += f"Please confirm your renewal:"

        keyboard = [
            [InlineKeyboardButton("✅ Confirm Renewal", callback_data=f"confirm_renewal_{service_id}_{package_id}_{cycle_id}_{xxx_addon}_{username}")],
            [InlineKeyboardButton("❌ Cancel", callback_data="back_to_menu")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]

        reply_markup = InlineKeyboardMarkup(keyboard)

        await context.bot.edit_message_text(
            chat_id=chat_id,
            message_id=message_id,
            text=confirmation_text,
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    except Exception as e:
        print(f"Error in process_direct_renewal: {e}")
        keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await context.bot.edit_message_text(
            chat_id=chat_id,
            message_id=message_id,
            text="❌ <b>Renewal Error</b>\n\nSomething went wrong. Please try again or contact support.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

async def show_services(context, chat_id, message_id, order_type):
    """Show available services"""
    services = package_manager.get_services()
    
    # Special handling for add_funds
    if order_type == "add_funds":
        # Only show the add_funds service
        service = package_manager.get_service_by_id("add_funds")
        if service:
            await show_billing_cycles(context, chat_id, message_id, order_type, service['id'])
            return
        else:
            keyboard = [[InlineKeyboardButton("« Back to Wallet", callback_data="wallet")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await context.bot.edit_message_text(
                chat_id=chat_id,
                message_id=message_id,
                text="❌ <b>Add Funds service not available</b>\n\nPlease contact support.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
            return
    
    # Filter out add_funds and any wallet_credit type services from regular selection
    services = [s for s in services if s['id'] != 'add_funds' and s.get('order_type') != 'wallet_credit']
    
    if not services:
        if order_type == "new_customer":
            keyboard = [[InlineKeyboardButton("« Back to Start", callback_data="back_to_start")]]
        else:
            keyboard = [[InlineKeyboardButton("« Back to Menu", callback_data="back_to_menu")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await context.bot.edit_message_text(
            chat_id=chat_id,
            message_id=message_id,
            text="❌ <b>No services available</b>\n\nNo services are currently available.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
        return
    
    keyboard = []
    for service in services:
        callback_data = f"select_service_{order_type}_{service['id']}"
        keyboard.append([InlineKeyboardButton(f"🔘 {service['name']}", callback_data=callback_data)])
    
    if order_type == "new_customer":
        keyboard.append([InlineKeyboardButton("« Back to Start", callback_data="back_to_start")])
    else:
        keyboard.append([InlineKeyboardButton("« Back to Menu", callback_data="back_to_menu")])
    
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    if order_type == "new_customer":
        title = "🆕 <b>Welcome New Customer!</b>\n\nChoose your service:"
    elif order_type == "renewal":
        title = "🔄 <b>Renewal Order</b>\n\nChoose your service:"
    else:
        title = "🛒 <b>New Order</b>\n\nChoose your service:"
    
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=title,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_billing_cycles(context, chat_id, message_id, order_type, service_id):
    """Show billing cycles for selected service"""
    service = package_manager.get_service_by_id(service_id)
    
    if not service:
        # Handle missing service
        keyboard = []
        if order_type == "new_customer":
            keyboard.append([InlineKeyboardButton("« Back to Services", callback_data="new_customer")])
        elif order_type == "add_funds":
            keyboard.append([InlineKeyboardButton("« Back to Wallet", callback_data="wallet")])
        else:
            keyboard.append([InlineKeyboardButton("« Back to Services", callback_data="new_order")])
        reply_markup = InlineKeyboardMarkup(keyboard)
        await context.bot.edit_message_text(
            chat_id=chat_id,
            message_id=message_id,
            text="❌ <b>Service Not Found</b>\n\nThe requested service could not be found.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
        return
    
    order_service_type = service.get('order_type', 'direct_cycles')
    keyboard = []
    cycle_list = ""
    promo_type = service.get('promo_type', 'cycle')

    if promo_type == 'cycle_connection':
        # Show each connection as a separate button for each billing cycle
        for cycle in service.get('billing_cycles', []):
            if not cycle.get('enabled', True):
                continue  # Skip disabled cycles
            for conn in cycle.get('connections', []):
                if not conn.get('enabled', True):
                    continue  # Skip disabled connections
                promo = promo_manager.get_promo(service['id'], cycle['cycle'], conn['count'])
                if promo:
                    price_text_btn = f"${promo['sale_price']:.2f} 🔥"
                    price_text_msg = f"<s>${promo['original_price']:.2f}</s> <b>${promo['sale_price']:.2f}</b> 🔥"
                else:
                    price_text_btn = f"${conn['base_price']:.2f}"
                    price_text_msg = f"${conn['base_price']:.2f}"
                button_label = f"📅 {cycle['name']} | 🔗 {conn['name']} - {price_text_btn}"
                callback_data = f"select_connection_{order_type}_{service['id']}_{cycle['cycle']}_{conn['count']}"
                keyboard.append([InlineKeyboardButton(button_label, callback_data=callback_data)])
                cycle_list += f"📅 <b>{cycle['name']}</b> | 🔗 {conn['name']} - {price_text_msg}\n"
    else:
        # Show each billing cycle as a button
        for cycle in service.get('billing_cycles', []):
            if not cycle.get('enabled', True):
                continue  # Skip disabled cycles
            promo = promo_manager.get_promo(service['id'], cycle['cycle'], None)
            if promo:
                price_text_btn = f"${promo['sale_price']:.2f} 🔥"
                price_text_msg = f"<s>${promo['original_price']:.2f}</s> <b>${promo['sale_price']:.2f}</b> 🔥"
            else:
                price_text_btn = f"${cycle['price']:.2f}"
                price_text_msg = f"${cycle['price']:.2f}"
            button_label = f"📅 {cycle['name']} - {price_text_btn}"
            if order_service_type == 'matrix':
                callback_data = f"select_matrix_cycle_{order_type}_{service['id']}_{cycle['cycle']}"
            else:
                callback_data = f"select_direct_cycle_{order_type}_{service['id']}_{cycle['cycle']}"
            keyboard.append([InlineKeyboardButton(button_label, callback_data=callback_data)])
            cycle_list += f"📅 <b>{cycle['name']}</b> - {price_text_msg}\n"
    
    # Add back button
    if order_type == "new_customer":
        keyboard.append([InlineKeyboardButton("« Back to Services", callback_data="new_customer")])
    elif order_type == "add_funds":
        keyboard.append([InlineKeyboardButton("« Back to Wallet", callback_data="wallet")])
    else:
        keyboard.append([InlineKeyboardButton("« Back to Services", callback_data="new_order")])
    
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    # Title/message
    if service_id == "add_funds":
        title = f"💰 <b>Add Funds to Wallet</b>\n\nChoose amount to add:\n\n{cycle_list}"
    else:
        setup_fee_text = ""
        if order_type != "renewal" and service.get('setup_fee', 0) > 0:
            setup_fee_text = f"\n💰 <b>Setup Fee:</b> ${service['setup_fee']:.2f} (one-time)"
        elif order_type == "renewal":
            setup_fee_text = f"\n💰 <b>Setup Fee:</b> $0.00 (waived for renewals)"
        title = f"🔘 <b>{service['name']}</b>\n\nChoose your billing cycle:\n\n{cycle_list}{setup_fee_text}"
    
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=title,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_connections(context, chat_id, message_id, order_type, service_id, cycle_id):
    """Show connection options for matrix services"""
    service = package_manager.get_service_by_id(service_id)
    cycle = package_manager.get_billing_cycle(service_id, cycle_id)
    
    if not service or not cycle or 'connections' not in cycle:
        return
    
    keyboard = []
    connections_list = ""
    
    for conn in cycle['connections']:
        if conn.get('enabled', True):
            callback_data = f"select_connection_{order_type}_{service_id}_{cycle_id}_{conn['count']}"
            keyboard.append([InlineKeyboardButton(f"🔗 {conn['name']} - ${conn['base_price']:.2f}", callback_data=callback_data)])
            connections_list += f"🔗 <b>{conn['name']}</b> - ${conn['base_price']:.2f}\n"
    
    keyboard.append([InlineKeyboardButton("« Back to Billing", callback_data=f"select_service_{order_type}_{service_id}")])
    
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    setup_fee_text = ""
    if order_type != "renewal" and service.get('setup_fee', 0) > 0:
        setup_fee_text = f"\n💰 <b>Setup Fee:</b> ${service['setup_fee']:.2f} (one-time)"
    elif order_type == "renewal":
        setup_fee_text = f"\n💰 <b>Setup Fee:</b> $0.00 (waived for renewals)"
    
    title = f"🔘 <b>{service['name']} - {cycle['name']}</b>\n\nChoose your connection count:\n\n{connections_list}{setup_fee_text}"
    
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=title,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_xxx_addon_matrix(context, chat_id, message_id, order_type, service_id, cycle_id, connection_count):
    """Show XXX addon option for matrix services"""
    service = package_manager.get_service_by_id(service_id)
    cycle = package_manager.get_billing_cycle(service_id, cycle_id)
    connection = package_manager.get_connection_option(service_id, cycle_id, connection_count)
    
    if not service or not cycle or not connection:
        return
    
    # Check if service supports XXX content
    xxx_support = service.get('xxx_support', False)
    if not xxx_support:
        await show_order_confirmation_matrix(context, chat_id, message_id, order_type, service_id, cycle_id, connection_count, False)
        return

    # --- PROMO LOGIC ---
    promo = promo_manager.get_promo(service['id'], cycle['cycle'], connection['count'])
    if promo:
        price_text = f"<s>${promo['original_price']:.2f}</s> <b>${promo['sale_price']:.2f}</b> 🔥"
    else:
        price_text = f"${connection['base_price']:.2f}"

    keyboard = [
        [InlineKeyboardButton("✅ Yes (Free)", callback_data=f"xxx_matrix_{order_type}_{service_id}_{cycle_id}_{connection_count}_yes")],
        [InlineKeyboardButton("❌ No", callback_data=f"xxx_matrix_{order_type}_{service_id}_{cycle_id}_{connection_count}_no")],
        [InlineKeyboardButton("« Back to Connections", callback_data=f"select_matrix_cycle_{order_type}_{service_id}_{cycle_id}")]
    ]
    
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    title = f"🔞 <b>XXX Content Add-on</b>\n\n"
    title += f"<b>Service:</b> {service['name']}\n"
    title += f"<b>Billing:</b> {cycle['name']}\n"
    title += f"<b>Connections:</b> {connection['name']}\n"
    title += f"<b>Base Price:</b> {price_text}\n\n"
    title += f"Would you like to add XXX content? (Free add-on)"
    
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=title,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_order_confirmation_direct(
    context,
    chat_id: int,
    message_id: int,
    order_type: str,
    service_id: str,
    cycle_id: str
):
    """Show order confirmation for direct cycle services."""
    service = package_manager.get_service_by_id(service_id)
    cycle = package_manager.get_billing_cycle(service_id, cycle_id)
    if not service or not cycle:
        return

    is_wallet_credit = service.get('order_type') == 'wallet_credit'
    promo = promo_manager.get_promo(service['id'], cycle['cycle'], None)
    if promo:
        base_price = promo['sale_price']
        original_price = promo['original_price']
        price_line = f"<s>${original_price:.2f}</s> <b>${base_price:.2f}</b> 🔥"
    else:
        base_price = cycle['price']
        price_line = f"${base_price:.2f}"

    setup_fee = 0 if order_type == "renewal" or is_wallet_credit else service.get('setup_fee', 0)
    total_price = base_price + setup_fee

    if is_wallet_credit:
        confirmation_text = (
            "💰 <b>Add Funds Confirmation</b>\n\n"
            f"<b>Amount to Add:</b> {price_line}\n\n"
            f"💳 <b>Payment Required:</b> ${total_price:.2f}\n\n"
            "📝 <b>Note:</b> Funds will be added to your wallet after payment verification.\n"
            "Wallet funds never expire and can be used for any service.\n\n"
            "Please confirm your order:"
        )
    else:
        confirmation_text = (
            ("🔄 <b>Renewal Confirmation</b>\n\n" if order_type == "renewal" else "📋 <b>Order Confirmation</b>\n\n") +
            f"<b>Service:</b> {service['name']}\n"
            f"<b>Billing Cycle:</b> {cycle['name']}\n\n"
            "💰 <b>Pricing Breakdown:</b>\n"
            f"• Base Price: {price_line}\n"
        )
        if setup_fee > 0:
            confirmation_text += f"• Setup Fee: ${setup_fee:.2f}\n"
        elif order_type == "renewal":
            confirmation_text += "• Setup Fee: $0.00 (waived for renewals)\n"
        confirmation_text += f"<b>Total: ${total_price:.2f}</b>\n\nPlease confirm your order:"

    keyboard = [
        [InlineKeyboardButton("✅ Confirm Order", callback_data=f"confirm_direct_{order_type}_{service_id}_{cycle_id}")],
        [InlineKeyboardButton(
            "📅 Change Amount" if is_wallet_credit else "📅 Change Billing",
            callback_data="add_funds" if is_wallet_credit else f"select_service_{order_type}_{service_id}"
        )],
        [InlineKeyboardButton(
            "❌ Cancel",
            callback_data="wallet" if is_wallet_credit else ("new_customer" if order_type == "new_customer" else "new_order")
        )]
    ]
    reply_markup = InlineKeyboardMarkup(keyboard)
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=confirmation_text,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_order_confirmation_matrix(
    context,
    chat_id: int,
    message_id: int,
    order_type: str,
    service_id: str,
    cycle_id: str,
    connection_count: int,
    xxx_addon: bool
):
    """Show order confirmation for matrix services."""
    service = package_manager.get_service_by_id(service_id)
    cycle = package_manager.get_billing_cycle(service_id, cycle_id)
    connection = package_manager.get_connection_option(service_id, cycle_id, connection_count)
    if not service or not cycle or not connection:
        return

    promo = promo_manager.get_promo(service['id'], cycle['cycle'], connection['count'])
    if promo:
        base_price = promo['sale_price']
        original_price = promo['original_price']
        price_line = f"<s>${original_price:.2f}</s> <b>${base_price:.2f}</b> 🔥"
    else:
        base_price = connection['base_price']
        price_line = f"${base_price:.2f}"

    setup_fee = 0 if order_type == "renewal" else service.get('setup_fee', 0)
    total_price = base_price + setup_fee

    confirmation_text = (
        ("🔄 <b>Renewal Confirmation</b>\n\n" if order_type == "renewal" else "📋 <b>Order Confirmation</b>\n\n") +
        f"<b>Service:</b> {service['name']}\n"
        f"<b>Billing Cycle:</b> {cycle['name']}\n"
        f"<b>Connections:</b> {connection['name']}\n"
    )
    if service.get('xxx_support', False):
        confirmation_text += f"<b>XXX Content:</b> {'Yes' if xxx_addon else 'No'}\n\n"
    else:
        confirmation_text += "\n"

    confirmation_text += (
        "💰 <b>Pricing Breakdown:</b>\n"
        f"• Base Price: {price_line}\n"
    )
    if setup_fee > 0:
        confirmation_text += f"• Setup Fee: ${setup_fee:.2f}\n"
    elif order_type == "renewal":
        confirmation_text += "• Setup Fee: $0.00 (waived for renewals)\n"
    confirmation_text += f"<b>Total: ${total_price:.2f}</b>\n\nPlease confirm your order:"

    keyboard = [
        [InlineKeyboardButton("✅ Confirm Order", callback_data=f"confirm_matrix_{order_type}_{service_id}_{cycle_id}_{connection_count}_{xxx_addon}")],
        [InlineKeyboardButton("🔗 Change Connections", callback_data=f"select_matrix_cycle_{order_type}_{service_id}_{cycle_id}")],
        [InlineKeyboardButton("📅 Change Billing", callback_data=f"select_service_{order_type}_{service_id}")],
        [InlineKeyboardButton("❌ Cancel Order", callback_data="new_customer" if order_type == "new_customer" else "new_order")]
    ]
    if service.get('xxx_support', False):
        keyboard.insert(1, [InlineKeyboardButton("🔞 Change XXX Option", callback_data=f"select_connection_{order_type}_{service_id}_{cycle_id}_{connection_count}")])

    reply_markup = InlineKeyboardMarkup(keyboard)
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=confirmation_text,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def show_payment_instructions_with_wallet(
    context,
    chat_id: int,
    message_id: int,
    order_id: int,
    service: dict,
    package: dict,
    cycle: dict,
    total_price: float,
    xxx_addon: bool,
    bot_payment_ref: str,
    user_id: int,
    order_type: str = "new"
):
    """Show payment instructions with wallet option."""
    is_add_funds = service.get('id') == 'add_funds'
    wallet_balance = 0.0
    amount_to_pay = total_price
    wallet_applied = 0.0

    if not is_add_funds:
        wallet_balance = db.get_wallet_balance(user_id)
        if wallet_balance > 0:
            if wallet_balance >= total_price:
                wallet_applied = total_price
                amount_to_pay = 0.0
            else:
                wallet_applied = wallet_balance
                amount_to_pay = total_price - wallet_balance

    cashapp_tag = service.get('cashapp_tag', '$Service918').replace('$', '')

    payment_text = (
        "💳 <b>Payment Instructions</b>\n\n"
        f"<b>Order ID:</b> #{order_id}\n"
        f"<b>Payment Reference:</b> <code>{bot_payment_ref}</code>\n"
        f"<b>Service:</b> {service['name']}\n"
    )
    if is_add_funds:
        payment_text += (
            f"<b>Add Funds Amount:</b> ${total_price:.2f}\n\n"
            "⚠️ <b>Note:</b> Wallet funds cannot be used for Add Funds purchases.\n\n"
        )
    else:
        if package and 'description' in package:
            payment_text += f"<b>Package:</b> {package['name']} ({package['description']})\n"
        else:
            payment_text += f"<b>Package:</b> {package['name']}\n"
        payment_text += f"<b>Billing:</b> {cycle['name']}\n"
        if service.get('xxx_support', False):
            payment_text += f"<b>XXX Content:</b> {'Included' if xxx_addon else 'Not Included'}\n"

    payment_text += f"\n💰 <b>Payment Details:</b>\n<b>Order Total:</b> ${total_price:.2f}\n"
    if not is_add_funds and wallet_balance > 0:
        payment_text += f"<b>Wallet Balance:</b> ${wallet_balance:.2f}\n"
        if wallet_applied > 0:
            payment_text += f"<b>Wallet Applied:</b> -${wallet_applied:.2f}\n"
        payment_text += f"<b>Amount to Pay:</b> ${amount_to_pay:.2f}\n"

    keyboard = []
    if not is_add_funds and amount_to_pay == 0:
        payment_text += (
            "\n✅ <b>Your wallet balance covers this order!</b>\n"
            "Click 'Pay with Wallet' to complete your order instantly.\n"
        )
        keyboard.append([InlineKeyboardButton("💰 Pay with Wallet", callback_data=f"pay_with_wallet_{order_id}")])
        keyboard.append([InlineKeyboardButton("❌ Cancel Order", callback_data=f"cancel_order_{order_id}")])
    else:
        payment_text += (
            "\n📱 <b>Cash App Payment Required:</b>\n"
            "1. Click 'Pay with Cash App' button below\n"
            f"2. Send exactly: <b>${amount_to_pay:.2f}</b>\n"
            f"3. In the note/memo, enter: <code>{bot_payment_ref}</code>\n"
            "4. Complete the payment\n"
        )
        if not is_add_funds and wallet_applied > 0:
            payment_text += f"\n💰 ${wallet_applied:.2f} from your wallet will be applied after payment verification.\n"
        cashapp_url = f"https://cash.app/{cashapp_tag}/{amount_to_pay:.2f}"
        keyboard.extend([
            [InlineKeyboardButton("💳 Pay with Cash App", url=cashapp_url)],
            [InlineKeyboardButton("✅ I've Sent Payment", callback_data=f"payment_sent_{order_id}")],
            [InlineKeyboardButton("❌ Cancel Order", callback_data=f"cancel_order_{order_id}")]
        ])
    keyboard.append([InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")])
    reply_markup = InlineKeyboardMarkup(keyboard)

    # Store wallet info in order for later processing
    if not is_add_funds and wallet_applied > 0:
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE orders SET details = ? WHERE id = ?",
            (f"wallet_applied:{wallet_applied}", order_id)
        )
        conn.commit()
        conn.close()

    try:
        await context.bot.edit_message_text(
            chat_id=chat_id,
            message_id=message_id,
            text=payment_text,
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
    except telegram.error.BadRequest as e:
        if "Message is not modified" in str(e):
            logging.info(f"Payment instructions already displayed for order {order_id}")
        else:
            raise

async def show_renewal_payment_instructions(context, chat_id, message_id, order_id, service, package, cycle, total_price, xxx_addon, username, bot_payment_ref):
    """Show payment instructions for account renewal with deeplink button"""
    cashapp_tag = service.get('cashapp_tag', '$Service918').replace('$', '')  # Remove $ for URL
    
    payment_text = f"💳 <b>Renewal Payment Instructions</b>\n\n"
    payment_text += f"<b>Order ID:</b> #{order_id}\n"
    payment_text += f"<b>Payment Reference:</b> <code>{bot_payment_ref}</code>\n"
    payment_text += f"<b>Renewing Account:</b> {username}\n"
    payment_text += f"<b>Service:</b> {service['name']}\n"
    payment_text += f"<b>Package:</b> {package['name']} ({package['description']})\n"
    payment_text += f"<b>Billing:</b> {cycle['name']}\n"
    
    if service.get('xxx_support', False):  # Only show XXX if service supports it
        payment_text += f"<b>XXX Content:</b> {'Included' if xxx_addon else 'Not Included'}\n"
    
    payment_text += f"<b>Total Amount:</b> ${total_price:.2f}\n\n"
    
    payment_text += f"📱 <b>Cash App Payment:</b>\n"
    payment_text += f"1. Click 'Pay with Cash App' button below\n"
    payment_text += f"2. Confirm the amount: <b>${total_price:.2f}</b>\n"
    payment_text += f"3. In the note/memo, enter: <code>{bot_payment_ref}</code>\n"
    payment_text += f"4. Complete the payment\n\n"
    
    payment_text += f"⏱️ <b>Processing Time:</b>\n"
    payment_text += f"• Payment verification: 1-5 minutes\n"
    payment_text += f"• Account renewal: Automatic\n"
    payment_text += f"• You'll receive confirmation here\n\n"
    
    payment_text += f"❗ <b>Important:</b>\n"
    payment_text += f"• Include <code>{bot_payment_ref}</code> in payment note\n"
    payment_text += f"• Send exact amount: ${total_price:.2f}\n"
    payment_text += f"• Your existing account will be extended\n\n"
    
    payment_text += f"💬 Need help? Contact support!"
    
    # Create Cash App deeplink
    cashapp_url = f"https://cash.app/{cashapp_tag}/{total_price:.2f}"
    
    keyboard = [
        [InlineKeyboardButton("💳 Pay with Cash App", url=cashapp_url)],
        [InlineKeyboardButton("✅ I've Sent Payment", callback_data=f"payment_sent_{order_id}")],
        [InlineKeyboardButton("📋 Check Payment Status", callback_data=f"check_payment_{order_id}")],
        [InlineKeyboardButton("❌ Cancel Renewal", callback_data=f"cancel_order_{order_id}")],
        [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
    ]
    
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    await context.bot.edit_message_text(
        chat_id=chat_id,
        message_id=message_id,
        text=payment_text,
        reply_markup=reply_markup,
        parse_mode='HTML'
    )

async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Start command handler with customer type selection"""
    user = update.effective_user
    print(f"🔍 /start command from user: {user.id} (@{user.username}) - {user.first_name}")
    
    # Check if user has a Telegram username
    if not user.username:
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        username_required_text = f"""❌ <b>Telegram Username Required</b>

Hello {user.first_name}!

To use this bot, you need to set up a Telegram username (@username).

<b>📝 How to set up your username:</b>

1. Open Telegram Settings
2. Tap on "Username" 
3. Create a unique username (letters, numbers, underscores)
4. Save your changes
5. Return here and type /start again

<b>Why do we need this?</b>
• Account security and verification
• Order tracking and support
• Account linking verification

Please set up your username and try again!"""
        
        await update.message.reply_text(username_required_text, parse_mode='HTML')
        return
    
    db.add_user(user.id, user.username or "")
    db.clear_user_state(user.id)
    
    # Run automatic pruning
    await prune_dead_accounts()
    
    bot_name = os.getenv("BOT_NAME", "Bot Management System")
    welcome_text = f"""🎬 <b>Welcome to {bot_name}</b>

Hello {user.first_name}! 

Let's get started:

<b>👤 User Login</b> - For those with active service.
<b>🆕 New Customer</b> - Order new service today!

Choose an option below:"""
    
    reply_markup = get_initial_keyboard()
    await update.message.reply_text(welcome_text, reply_markup=reply_markup, parse_mode='HTML')

async def about_handler(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        with open("about.json", "r", encoding="utf-8") as f:
            data = json.load(f)
    except Exception:
        await update.message.reply_text("⚠️ About information is unavailable.")
        return

    message = "ℹ️ <b>About Our Services</b>\n\n"
    if "info" in data:
        message += f"{data['info']}\n\n"
    if "services" in data:
        message += "<b>Services:</b>\n"
        for svc in data["services"]:
            message += f"• <b>{svc['name']}</b>: {svc.get('description','')}\n"
            if svc.get("setup"):
                message += f"  <i>Setup:</i> {svc['setup']}\n"
        message += "\n"
    if "instructions" in data:
        message += f"<b>Instructions:</b> {data['instructions']}\n\n"
    if "support_link" in data:
        message += f"💬 <a href=\"{data['support_link']}\">Support Group</a>"

    await update.message.reply_text(message, parse_mode="HTML", disable_web_page_preview=True)

async def debug_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Debug command to check admin status"""
    user_id = update.effective_user.id
    
    debug_info = f"""🔍 <b>Debug Information</b>

<b>Your Telegram ID:</b> {user_id}
<b>Configured Admin IDs:</b> {ADMIN_USER_IDS}
<b>Is Admin Check:</b> {db.is_admin(user_id)}
<b>ID Match:</b> {user_id in ADMIN_USER_IDS}

<b>Database Status:</b>"""
    
    conn = sqlite3.connect(db.db_path)
    cursor = conn.cursor()
    cursor.execute("SELECT user_id, telegram_username, is_admin FROM users")
    all_users = cursor.fetchall()
    conn.close()
    
    for user_record in all_users:
        debug_info += f"\nUser: {user_record[0]} (@{user_record[1]}) Admin: {user_record[2]}"
    
    await update.message.reply_text(debug_info, parse_mode='HTML')

async def broadcast_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if update.effective_user.id not in ADMIN_USER_IDS:
        await update.message.reply_text("❌ Only admins can broadcast messages.")
        return

    if not context.args:
        await update.message.reply_text("Usage: <code>/broadcast your message here</code>", parse_mode="HTML")
        return

    msg = " ".join(context.args)
    conn = sqlite3.connect(db.db_path)
    cursor = conn.cursor()
    cursor.execute("SELECT user_id FROM users")
    all_users = [row[0] for row in cursor.fetchall()]
    conn.close()

    sent_count = 0
    for uid in all_users:
        try:
            await context.bot.send_message(uid, f"📢 <b>Announcement:</b>\n\n{msg}", parse_mode="HTML")
            sent_count += 1
        except Exception as e:
            # Log or ignore failures (user blocked bot, etc.)
            pass

    await update.message.reply_text(f"✅ Broadcast sent to {sent_count} user(s).")

async def promo_list_command(update, context):  
    if update.effective_user.id not in ADMIN_USER_IDS:  
        await update.message.reply_text("❌ Only admins can view promos.")  
        return  
  
    promos = promo_manager.promos  
    if not promos:  
        await update.message.reply_text("No active promos.")  
        return  
  
    # You may need to adjust these lookups to match your package manager structure  
    msg = "<b>Active Promos:</b>\n\n"  
    now = datetime.now()  
    for key, promo in promos.items():  
        # Key format: service_id|cycle_id|connection_count  
        parts = key.split('|')  
        service_id, cycle_id, conn_count = parts[0], parts[1], parts[2]  
        service = package_manager.get_service_by_id(service_id)  
        service_name = service['name'] if service else service_id  
        cycle = package_manager.get_billing_cycle(service_id, cycle_id)  
        cycle_name = cycle['name'] if cycle else cycle_id  
  
        # Format time left  
        time_left = promo['end_time'] - now  
        minutes_left = int(time_left.total_seconds() // 60)  
        if minutes_left < 1:  
            time_left_str = "<1 min"  
        else:  
            time_left_str = f"{minutes_left} min"  
  
        msg += f"• <b>{service_name}</b> | <b>{cycle_name}</b>"  
        if conn_count != "None":  
            msg += f" | <b>{conn_count} Connections</b>"  
        msg += (  
            f"\n  <s>${promo['original_price']:.2f}</s> <b>${promo['sale_price']:.2f}</b> 🔥"  
            f"  (<i>{time_left_str} left</i>)\n\n"  
        )  
  
    await update.message.reply_text(msg, parse_mode="HTML")

async def promoclear_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    args = context.args
    if len(args) < 2:
        await update.message.reply_text(
            "Usage:\n"
            "/promoclear <service_id> <cycle_id> [connection_count]\n\n"
            "Example for Service1: /promoclear service1 monthly 5\n"
            "Example for Service2/3: /promoclear service2 monthly"
        )
        return

    service_id = args[0]
    cycle_id = args[1]
    connection_count = int(args[2]) if len(args) > 2 else None

    promo_manager.clear_promo(service_id, cycle_id, connection_count)

    msg = f"✅ Promo cleared for Service: {service_id}, Cycle: {cycle_id}"
    if connection_count is not None:
        msg += f", Connections: {connection_count}"

    print("PROMOCLEAR MSG:", msg)

    await update.message.reply_text(msg)

async def button_callback(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Handle button callbacks"""
    query = update.callback_query
    
    user = query.from_user
    user_id = user.id
    if is_admin_only_mode() and user_id not in ADMIN_USER_IDS:
        await query.edit_message_text(
            "🚧 <b>The bot is currently in admin-only mode for maintenance or updates.</b>\n\n"
            "Please try again later.",
            parse_mode='HTML'
        )
        return
    
    # Check if user has a Telegram username for most actions
    if not user.username and not query.data.startswith("back_to_start"):
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        username_required_text = f"""❌ <b>Telegram Username Required</b>

Hello {user.first_name}!

To use this bot, you need to set up a Telegram username (@username).

<b>📝 How to set up your username:</b>

1. Open Telegram Settings
2. Tap on "Username" 
3. Create a unique username (letters, numbers, underscores)
4. Save your changes
5. Return here and type /start again

<b>Why do we need this?</b>
• Account security and verification
• Order tracking and support
• Account linking verification

Please set up your username and try again!"""
        
        await query.edit_message_text(username_required_text, parse_mode='HTML')
        return
    
    data = query.data
    user_id = query.from_user.id
    chat_id = query.message.chat_id
    message_id = query.message.message_id
    
    print(f"🔍 Button callback: {data} from user {user_id}")
    
    if data == "back_to_start":
        db.clear_user_state(user_id)
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        welcome_text = f"""🎬 <b>Welcome to {bot_name}</b>

Hello {query.from_user.first_name}! 

Let's get started:

<b>👤 User Login</b> - For those with active service.
<b>🆕 New Customer</b> - Order new service today!

Choose an option below:"""
        
        reply_markup = get_initial_keyboard()
        await query.edit_message_text(welcome_text, reply_markup=reply_markup, parse_mode='HTML')
    
    elif data == "subscriber":
        db.clear_user_state(user_id)
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        welcome_text = f"""🎬 <b>User Login Portal</b>

Welcome back {query.from_user.first_name}! 

This bot helps you manage your service accounts securely:
• Link your service accounts to access information
• Check account details and expiry dates
• Place new subscription orders
• Renew existing subscriptions (within 10 days of expiry)
• View your order history

<b>Security Notice:</b> You can only access accounts that you have linked and verified.

Choose an option below:"""
        
        reply_markup = get_main_menu_keyboard(user_id)
        await query.edit_message_text(welcome_text, reply_markup=reply_markup, parse_mode='HTML')
    
    elif data == "new_customer":
        db.clear_user_state(user_id)
        await show_services(context, chat_id, message_id, "new_customer")
    
    elif data == "new_order":
        db.clear_user_state(user_id)
        await show_services(context, chat_id, message_id, "new_order")
    
    elif data == "back_to_menu":
        db.clear_user_state(user_id)
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        welcome_text = f"""🎬 <b>User Login Portal</b>

Welcome back {query.from_user.first_name}! 

This portal helps you manage your service accounts securely:
• Link your panel accounts to access information
• Check account details and expiry dates
• Place new subscription orders
• Renew existing subscriptions (within 10 days of expiry)
• View your order history

<b>Security Notice:</b> You can only access accounts that you have linked and verified.

Choose an option below:"""
    
        reply_markup = get_main_menu_keyboard(user_id)
        await query.edit_message_text(welcome_text, reply_markup=reply_markup, parse_mode='HTML')
    
    # --- Support Ticket System: User Side ---
    elif data == "support_tickets":
        await query.edit_message_text(
            text="🆘 Support Ticket Menu:\nPlease choose an option:",
            reply_markup=InlineKeyboardMarkup([
                [InlineKeyboardButton("➕ New Ticket", callback_data="ticket_new")],
                [InlineKeyboardButton("📋 My Tickets", callback_data="ticket_menu")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ])
        )
    elif data == "ticket_menu":
        await user_ticket_menu(update, context)
    elif data.startswith("ticket_view_"):
        ticket_id = data.split("_", 2)[2]
        await user_ticket_view(update, context, ticket_id)
    elif data.startswith("ticket_reply_"):
        ticket_id = data.split("_", 2)[2]
        await user_ticket_reply_start(update, context, ticket_id)
    elif data.startswith("ticket_close_"):
        ticket_id = data.split("_", 2)[2]
        await user_ticket_close(update, context, ticket_id)
    elif data == "ticket_new":
        await user_ticket_new_start(update, context)
    elif data.startswith("ticket_priority_"):
        priority = data.split("_", 2)[2].capitalize()
        await handle_priority_selection(update, context, priority)
    elif data.startswith("ticket_refresh_"):
        ticket_id = data.split("_", 2)[2]
        await user_ticket_view(update, context, ticket_id)

    # --- Support Ticket System: Admin Side ---
    elif data == "admin_ticket_menu":
        await admin_ticket_menu(update, context)
    elif data.startswith("admin_ticket_view_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_view(update, context, ticket_id)
    elif data.startswith("admin_ticket_reply_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_reply_start(update, context, ticket_id)
    elif data.startswith("admin_ticket_close_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_close(update, context, ticket_id)
    elif data.startswith("admin_ticket_delete_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_delete(update, context, ticket_id)
    elif data.startswith("admin_ticket_reopen_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_reopen(update, context, ticket_id)
    elif data.startswith("admin_ticket_refresh_"):
        ticket_id = data.split("_", 3)[3]
        await admin_ticket_view(update, context, ticket_id)

    elif data.startswith("select_service_"):
        parts = data.split("_")
        if len(parts) >= 4:
            service_id = parts[-1]
            order_type = "_".join(parts[2:-1])
            await show_billing_cycles(context, chat_id, message_id, order_type, service_id)
    
    elif data.startswith("select_package_"):
        parts = data.split("_")
        if len(parts) >= 5:
            package_id = parts[-1]
            service_id = parts[-2]
            order_type = "_".join(parts[2:-2])
            await show_billing_cycles(context, chat_id, message_id, order_type, service_id, package_id)
    
    elif data.startswith("select_cycle_"):
        parts = data.split("_")
        if len(parts) >= 6:
            cycle_id = parts[-1]
            package_id = parts[-2]
            service_id = parts[-3]
            order_type = "_".join(parts[2:-3])
            await show_xxx_addon(context, chat_id, message_id, order_type, service_id, package_id, cycle_id)
    
    elif data.startswith("xxx_addon_"):
        parts = data.split("_")
        if len(parts) >= 7:
            xxx_choice = parts[-1]
            cycle_id = parts[-2]
            package_id = parts[-3]
            service_id = parts[-4]
            order_type = "_".join(parts[2:-4])
            xxx_addon = xxx_choice == "yes"
            await show_order_confirmation(context, chat_id, message_id, order_type, service_id, package_id, cycle_id, xxx_addon)

    elif data.startswith("confirm_order_"):
        parts = data.split("_")
        if len(parts) >= 7:
            xxx_addon = parts[-1] == "True"
            cycle_id = parts[-2]
            package_id = parts[-3]
            service_id = parts[-4]
            order_type = "_".join(parts[2:-4])
            
            service = package_manager.get_service_by_id(service_id)
            # Simple package and cycle retrieval
            package = {"id": package_id, "name": f"Package {package_id}"}
            cycle = package_manager.get_billing_cycle(service_id, cycle_id)
            
            if service and cycle:
                # Generate username and password for new orders
                username, password = generate_username_password()
                
                base_price = cycle['price']
                setup_fee = 0 if order_type == "renewal" else service.get('setup_fee', 0)
                total_price = base_price + setup_fee
                
                # Create order with payment tracking
                order_id, bot_payment_ref = db.add_order_with_payment(
                    user_id, query.from_user.username, order_type, service_id, service['name'],
                    package_id, package['name'], cycle['cycle'], cycle.get('panel_package_id', ''),
                    base_price, xxx_addon, setup_fee, total_price, service.get('cashapp_tag', '$Service918'),
                    username, password, f"XXX: {'Yes' if xxx_addon else 'No'}"
                )
                
                await show_payment_instructions_with_wallet(context, chat_id, message_id, order_id, service, package, cycle, total_price, xxx_addon, bot_payment_ref, user_id, order_type)
    
    elif data.startswith("confirm_renewal_"):
        parts = data.split("_")
        if len(parts) >= 7:
            username = parts[-1]
            xxx_addon = parts[-2] == "True"
            cycle_id = parts[-3]
            package_id = parts[-4]
            service_id = parts[-5]
            
            service = package_manager.get_service_by_id(service_id)
            # For renewals, create appropriate package objects
            if service.get('order_type') == 'direct_cycles':
                package = {
                    'id': service['id'],
                    'name': service['name'],
                    'description': f'Renewal for {service["name"]}'
                }
            else:
                # For matrix services, we'd need proper package lookup
                package = {"id": package_id, "name": f"Package {package_id}", "description": "Renewal"}
            
            cycle = package_manager.get_billing_cycle(service_id, cycle_id)
            
            if service and cycle:
                base_price = cycle['price']
                setup_fee = 0  # No setup fee for renewals
                total_price = base_price + setup_fee
                
                # Create renewal order with payment tracking
                order_id, bot_payment_ref = db.add_order_with_payment(
                    user_id, query.from_user.username, "renewal", service_id, service['name'],
                    package_id, package['name'], cycle['cycle'], cycle.get('panel_package_id', ''),
                    base_price, xxx_addon, setup_fee, total_price, service.get('cashapp_tag', '$Service918'),
                    username, "", f"Renewal for {username}, XXX: {'Yes' if xxx_addon else 'No'}"
                )
                
                await show_payment_instructions_with_wallet(
                    context, chat_id, message_id, order_id, service, package, cycle, total_price, xxx_addon, bot_payment_ref, user_id, "renewal"
                )

    elif data.startswith("payment_sent_"):
        order_id = int(data.split("_")[-1])
        order_data = db.get_order_by_id(order_id)
        
        if order_data:
            bot_payment_ref = order_data['notes']  # Bot reference is in notes field
            payment_amount = order_data['payment_amount']
            
            await query.edit_message_text(
                f"✅ <b>Payment Confirmation Received</b>\n\n"
                f"<b>Order ID:</b> #{order_id}\n"
                f"<b>Payment Reference:</b> <code>{bot_payment_ref}</code>\n"
                f"<b>Amount:</b> ${payment_amount:.2f}\n\n"
                f"Thank you! Your payment confirmation has been received.\n\n"
                f"⏱️ <b>Next Steps:</b>\n"
                f"• Payment verification: 1-5 minutes\n"
                f"• Account creation: Automatic after verification\n"
                f"• You'll receive login details here\n\n"
                f"💬 If you don't receive your account within 10 minutes, please contact support.",
                parse_mode='HTML'
            )
        else:
            await query.edit_message_text("❌ Order not found.")
    
    elif data.startswith("cancel_order_"):
        order_id = int(data.split("_")[-1])
        db.update_order_status(order_id, "cancelled", "Cancelled by user")
        
        keyboard = [
            [InlineKeyboardButton("🛒 New Order", callback_data="new_order")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            f"❌ <b>Order Cancelled</b>\n\n"
            f"Order #{order_id} has been cancelled.\n\n"
            f"You can place a new order anytime.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
    
    elif data.startswith("pay_with_wallet_"):
        order_id = int(data.split("_")[-1])
        order_data = db.get_order_by_id(order_id)
        
        if not order_data:
            await query.edit_message_text("❌ Order not found.", parse_mode='HTML')
            return
        
        # Now we can access fields directly
        order_type = order_data['order_type']
        total_price = float(order_data['total_price'])
        service_name = order_data['service_name']
        username = order_data['username']  # for renewals
        service_id = order_data['service_id']
        panel_package_id = order_data['panel_package_id']
        
        # Deduct from wallet
        if db.deduct_wallet_funds(user_id, total_price, f"Payment for Order #{order_id} - {service_name}", order_id):
            # Update order status
            db.update_order_status(order_id, "completed", "Paid with wallet funds")
            
            # Process based on order type
            if order_type == "renewal":
                # For renewals, we need to process the renewal in the panel
                # Get panel for this service
                panel = panel_manager.get_panel_for_service(service_id)
                
                if panel and username:
                    # Process the renewal
                    notes = f"Bot Order #{order_id} - Wallet payment"
                    
                    renewal_result = panel.renew_user_account(
                        username=username,
                        package_id=panel_package_id,
                        duration_days=30,  # Default to 30 days
                        notes=notes
                    )
                    
                    if renewal_result.get('success'):
                        await query.edit_message_text(
                            f"✅ <b>Renewal Successful!</b>\n\n"
                            f"<b>Order #{order_id}</b> has been paid using your wallet funds.\n"
                            f"Amount: ${total_price:.2f}\n\n"
                            f"Account <code>{username}</code> has been renewed!\n"
                            f"New expiry: {renewal_result.get('new_expiry', 'N/A')}\n\n"
                            f"Your new wallet balance: ${db.get_wallet_balance(user_id):.2f}",
                            parse_mode='HTML'
                        )
                    else:
                        # Refund if renewal failed
                        db.add_wallet_funds(user_id, total_price, f"Refund for failed renewal order #{order_id}")
                        db.update_order_status(order_id, "failed", f"Panel renewal failed: {renewal_result.get('error')}")
                        
                        await query.edit_message_text(
                            f"❌ <b>Renewal Failed</b>\n\n"
                            f"The renewal could not be processed.\n"
                            f"Error: {renewal_result.get('error', 'Unknown error')}\n\n"
                            f"Your wallet has been refunded ${total_price:.2f}",
                            parse_mode='HTML'
                        )
                else:
                    await query.edit_message_text(
                        "❌ <b>Configuration Error</b>\n\n"
                        "Could not process renewal. Please contact support.",
                        parse_mode='HTML'
                    )
            else:
                # For new orders, create account in panel
                # Check if this is an add_funds order
                if service_id == "add_funds":
                    # Credit wallet with the amount
                    credit_amount = float(order_data['base_price'])
                    db.add_wallet_funds(user_id, credit_amount, f"Add Funds - Order #{order_id}", order_id)
                    
                    await query.edit_message_text(
                        f"✅ <b>Funds Added Successfully!</b>\n\n"
                        f"<b>Order #{order_id}</b> completed.\n"
                        f"Amount added: ${credit_amount:.2f}\n\n"
                        f"Your new wallet balance: ${db.get_wallet_balance(user_id):.2f}",
                        parse_mode='HTML'
                    )
                else:
                    # Regular new order - create account
                    panel = panel_manager.get_panel_for_service(service_id)

                    if panel:
                        # Get order details
                        new_username = order_data['username']
                        new_password = order_data['password']
                        package_id = order_data['panel_package_id']
                        
                        # Create account using the new method
                        create_result = panel.create_user_account(
                            username=new_username,
                            password=new_password,
                            package_id=package_id,
                            duration_days=30,
                            max_connections=1,
                            notes=f"Bot Order #{order_id} - Wallet payment"
                        )
                        
                        if create_result.get('success'):
                            # Update order with the ACTUAL credentials returned by panel
                            actual_username = create_result.get('username', new_username)
                            actual_password = create_result.get('password', new_password)
                            
                            # Update order status
                            db.update_order_status(order_id, "completed", "Account created successfully")
                            
                            # AUTO-LINK: Automatically link the account to the user
                            auto_link_success = False
                            conn = sqlite3.connect(db.db_path)
                            cursor = conn.cursor()
                            try:
                                # Check if link already exists
                                cursor.execute("""
                                    SELECT id FROM account_links 
                                    WHERE telegram_user_id = ? AND panel_username = ?
                                """, (user_id, actual_username))
                                
                                existing_link = cursor.fetchone()
                                
                                if not existing_link:
                                    # Create automatic link with approved status
                                    cursor.execute("""
                                        INSERT INTO account_links (telegram_user_id, panel_username, service_name, status, verification_code, linked_at, approved_by, created_at)
                                        VALUES (?, ?, ?, 'approved', ?, CURRENT_TIMESTAMP, 0, CURRENT_TIMESTAMP)
                                    """, (user_id, actual_username, service_name, f"AUTO_ORDER_{order_id}"))
                                    
                                    conn.commit()
                                    auto_link_success = True
                                    logging.info(f"AUTO-LINK: Account {actual_username} ({service_name}) automatically linked to user {user_id} from order #{order_id}")
                                else:
                                    logging.info(f"AUTO-LINK: Account {actual_username} already linked to user {user_id}")
                            except Exception as e:
                                logging.error(f"AUTO-LINK: Error linking account {actual_username}: {e}")
                                conn.rollback()
                            finally:
                                conn.close()
                            
                            # Get panel URL info for the message
                            panel_info = ""
                            if hasattr(panel, 'panel_url'):
                                base_url = panel.panel_url.replace(':25500', '').replace('http://', '').replace('https://', '')
                                panel_info = f"\n\n📡 <b>Panel URL:</b> <code>{base_url}</code>"
                            
                            # Send credentials to user with ACTUAL credentials
                            credentials_message = f"""✅ <b>Order Completed!</b>

                        <b>Order ID:</b> #{order_id}
                        <b>Service:</b> {service_name}
                        <b>Package:</b> {order_data['package_name']}

                        🔑 <b>Your Login Details:</b>

                        <b>Username:</b> <code>{actual_username}</code>
                        <b>Password:</b> <code>{actual_password}</code>

                        <b>Important:</b>
                        • Save these credentials securely
                        • Account has been automatically linked to your Telegram
                        • You can view this account in "My Accounts"
                        • Renewal available within 10 days of expiry

                        Thank you for your order! 🎉"""
                            
                            # Create navigation buttons
                            keyboard = [
                                [
                                    InlineKeyboardButton("📋 Copy Username", callback_data=f"copy_user_{actual_username}"),
                                    InlineKeyboardButton("📋 Copy Password", callback_data=f"copy_pass_{actual_username}")
                                ],
                                [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
                            ]
                            reply_markup = InlineKeyboardMarkup(keyboard)
                            
                            await query.edit_message_text(
                                credentials_message,
                                reply_markup=reply_markup,
                                parse_mode='HTML'
                            )
                        else:
                            # Account creation failed - refund
                            db.add_wallet_funds(user_id, total_price, f"Refund for failed order #{order_id}")
                            db.update_order_status(order_id, "failed", f"Account creation failed: {create_result.get('error', 'Unknown error')}")
                            
                            await query.edit_message_text(
                                f"❌ <b>Account Creation Failed</b>\n\n"
                                f"Order #{order_id} could not be completed.\n"
                                f"Error: {create_result.get('error', 'Unknown error')}\n\n"
                                f"Your wallet has been refunded ${total_price:.2f}\n\n"
                                f"Please try again or contact support.",
                                parse_mode='HTML'
                            )
                    else:
                        # No panel configured
                        await query.edit_message_text(
                            f"❌ <b>Configuration Error</b>\n\n"
                            f"Panel not configured for {service_name}.\n"
                            f"Please contact support.",
                            parse_mode='HTML'
                        )
        else:
            await query.edit_message_text(
                "❌ <b>Payment Failed</b>\n\n"
                "Insufficient wallet balance. Please add funds or use another payment method.",
                parse_mode='HTML'
            )

    elif data == "wallet":
        balance = db.get_wallet_balance(user_id)
        transactions = db.get_wallet_transactions(user_id, 5)
        
        # Get auto-renewal settings
        auto_renewal_settings = db.get_auto_renewal_settings(user_id)
        auto_renewal_status = "Disabled"
        if auto_renewal_settings and auto_renewal_settings[0]:  # enabled
            timing = auto_renewal_settings[1]
            if timing == "immediate":
                auto_renewal_status = "Pay Immediately"
            elif timing == "last_day":
                auto_renewal_status = "Pay on Last Day"
        
        wallet_text = f"💰 <b>Your Wallet</b>\n\n"
        wallet_text += f"<b>Current Balance:</b> ${balance:.2f}\n"
        wallet_text += f"<b>Auto-Renewal:</b> {auto_renewal_status}\n\n"
        
        if transactions:
            wallet_text += "<b>Recent Transactions:</b>\n"
            for trans_type, amount, balance_after, description, created_at in transactions:
                try:
                    trans_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
                except:
                    trans_date = created_at[:16] if created_at else "Unknown"
                
                symbol = "+" if trans_type == "credit" else "-"
                wallet_text += f"\n{symbol}${amount:.2f} - {description}\n"
                wallet_text += f"  Balance: ${balance_after:.2f} • {trans_date}\n"
        else:
            wallet_text += "\n<i>No transactions yet</i>\n"
        
        wallet_text += "\n💡 <b>Tips:</b>\n"
        wallet_text += "• Add funds to your wallet for quick payments\n"
        wallet_text += "• Wallet funds can be used for orders and renewals\n"
        wallet_text += "• Unused funds never expire"
        
        keyboard = [
            [InlineKeyboardButton("➕ Add Funds", callback_data="add_funds")],
            [InlineKeyboardButton("🔄 Auto-Renewal Settings", callback_data="auto_renewal_settings")],
            [InlineKeyboardButton("📜 Transaction History", callback_data="wallet_history")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(wallet_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "add_funds":
        # Show the Add Funds service specifically
        service = package_manager.get_service_by_id("add_funds")
        
        if service and service.get('billing_cycles'):
            keyboard = []
            cycle_list = ""
            
            for cycle in service['billing_cycles']:
                callback_data = f"select_direct_cycle_add_funds_add_funds_{cycle['cycle']}"
                keyboard.append([InlineKeyboardButton(f"💵 {cycle['name']}", callback_data=callback_data)])
                cycle_list += f"💵 <b>{cycle['name']}</b>\n"
            
            keyboard.append([InlineKeyboardButton("« Back to Wallet", callback_data="wallet")])
            
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            title = f"💰 <b>Add Funds to Wallet</b>\n\n"
            title += f"Choose amount to add:\n\n{cycle_list}\n"
            title += f"💡 <b>Note:</b> Funds added to your wallet never expire and can be used for any service orders or renewals."
            
            await query.edit_message_text(
                title,
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            await query.edit_message_text(
                "❌ Add Funds service not configured properly.",
                parse_mode='HTML'
            )

    elif data == "wallet_history":
        transactions = db.get_wallet_transactions(user_id, 20)
        
        history_text = "📜 <b>Wallet Transaction History</b>\n\n"
        
        if transactions:
            for trans_type, amount, balance_after, description, created_at in transactions:
                try:
                    trans_date = datetime.fromisoformat(created_at).strftime("%Y-%m-%d %H:%M")
                except:
                    trans_date = created_at[:19] if created_at else "Unknown"
                
                symbol = "+" if trans_type == "credit" else "-"
                history_text += f"{trans_date}\n"
                history_text += f"{symbol}${amount:.2f} - {description}\n"
                history_text += f"Balance: ${balance_after:.2f}\n\n"
        else:
            history_text += "<i>No transactions found</i>"
        
        keyboard = [
            [InlineKeyboardButton("💰 Back to Wallet", callback_data="wallet")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(history_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "dashboard_menu":
        await dashboard_command(update, context)
    
    elif data == "my_accounts":
        linked_accounts = db.get_linked_accounts(user_id)
        
        if not linked_accounts:
            keyboard = [
                [InlineKeyboardButton("🔗 Link Account", callback_data="link_account")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                "📱 <b>My Accounts</b>\n\nYou don't have any linked accounts yet.\n\nLink an existing account to access your subscription details.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            keyboard = []
            account_list = ""
            
            for i, (username, service_name) in enumerate(linked_accounts, 1):
                # Dynamic service emoji based on service
                service_emoji = "📱"  # Default
                service = package_manager.get_service_by_name(service_name)
                if service:
                    for service_num in range(1, 6):
                        if service['id'] == f"service{service_num}":
                            emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                            service_emoji = emoji_map.get(service_num, "📱")
                            break
                
                account_list += f"{i}. {service_emoji} {username} ({service_name})\n"
                keyboard.append([InlineKeyboardButton(f"{service_emoji} {username} ({service_name})", callback_data=f"check_account_{username}")])
            
            keyboard.extend([
                [InlineKeyboardButton("🔗 Link Another Account", callback_data="link_account")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ])
            
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                f"📱 <b>My Linked Accounts</b>\n\n{account_list}\nSelect an account to view details:",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )

    elif data == "auto_renewal_settings":
        settings = db.get_auto_renewal_settings(user_id)
        current_setting = "manual"
        if settings and settings[0]:  # enabled
            current_setting = settings[1]
        
        # Create status indicators
        immediate_check = "✅" if current_setting == "immediate" else "⚪"
        last_day_check = "✅" if current_setting == "last_day" else "⚪"
        manual_check = "✅" if current_setting == "manual" else "⚪"
        
        settings_text = f"🔄 <b>Auto-Renewal Settings</b>\n\n"
        settings_text += f"Configure how your renewals are handled when you have sufficient wallet funds.\n\n"
        settings_text += f"<b>Current Setting:</b> "
        
        if current_setting == "immediate":
            settings_text += "Pay Immediately\n"
            settings_text += "➜ Renewals are paid as soon as they become available (within 10 days of expiry)"
        elif current_setting == "last_day":
            settings_text += "Pay on Last Day\n"
            settings_text += "➜ Renewals are paid on the expiration day"
        else:
            settings_text += "Manual Renewal\n"
            settings_text += "➜ You must manually approve each renewal"
        
        settings_text += "\n\n<b>Select your preference:</b>"
        
        keyboard = [
            [InlineKeyboardButton(f"{immediate_check} Pay Immediately", callback_data="auto_renewal_immediate")],
            [InlineKeyboardButton(f"{last_day_check} Pay on Last Day", callback_data="auto_renewal_last_day")],
            [InlineKeyboardButton(f"{manual_check} Manual Renewal", callback_data="auto_renewal_manual")],
            [InlineKeyboardButton("💰 Back to Wallet", callback_data="wallet")]
        ]
        
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(settings_text, reply_markup=reply_markup, parse_mode='HTML')

    # Handle setting changes
    elif data.startswith("auto_renewal_"):
        setting = data.replace("auto_renewal_", "")
        
        if setting in ["immediate", "last_day", "manual"]:
            enabled = setting != "manual"
            db.set_auto_renewal_setting(user_id, enabled, setting)
            
            confirmation_text = "✅ <b>Auto-Renewal Updated</b>\n\n"
            
            if setting == "immediate":
                confirmation_text += "Your renewals will be automatically paid from your wallet as soon as they become available (within 10 days of expiry).\n\n"
                confirmation_text += "⚡ You'll be notified when payments are processed."
            elif setting == "last_day":
                confirmation_text += "Your renewals will be automatically paid from your wallet on the expiration day.\n\n"
                confirmation_text += "⏰ You'll be notified 24 hours before and when payment is processed."
            else:
                confirmation_text += "Auto-renewal is now disabled. You'll need to manually renew your accounts.\n\n"
                confirmation_text += "📱 You'll still receive renewal reminders."
            
            keyboard = [[InlineKeyboardButton("✅ OK", callback_data="auto_renewal_settings")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(confirmation_text, reply_markup=reply_markup, parse_mode='HTML')       
    
    elif data.startswith("check_account_"):
            username = data.replace("check_account_", "")
            
            if not db.can_access_account(user_id, username):
                await query.edit_message_text("❌ <b>Access Denied</b>\n\nYou don't have permission to access this account.", parse_mode='HTML')
                return
            
            # Get the service for this account
            service_name = db.get_account_service(user_id, username)
            service = package_manager.get_service_by_name(service_name)
            
            if not service:
                # Try to find service by legacy mapping
                for s in package_manager.get_services():
                    # Map based on environment variables
                    for service_num in range(1, 6):
                        env_name = os.getenv(f"SERVICE{service_num}_NAME", "")
                        if s['name'] == env_name and (
                            (service_name in ["Service1", "Service2", "Service3"] and service_name.lower() in env_name.lower()) or
                            service_name == env_name
                        ):
                            service = s
                            service_name = s['name']  # Update to current name
                            break
                    if service:
                        break
            
            if not service:
                keyboard = [[InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")]]
                reply_markup = InlineKeyboardMarkup(keyboard)
                await query.edit_message_text("❌ <b>Service Configuration Error</b>\n\nService not found.", reply_markup=reply_markup, parse_mode='HTML')
                return
            
            service_id = service['id']
            
            # Show loading message
            await query.edit_message_text(
                f"🔍 <b>Checking Account Status...</b>\n\n"
                f"👤 Account: {username}\n"
                f"🔘 Service: {service_name}\n\n"
                f"⏳ Connecting to panel...\n"
                f"📊 Retrieving account information...",
                parse_mode='HTML'
            )
            
            # Add small delay to show loading
            import asyncio
            await asyncio.sleep(1)
            
            # Use the correct panel for this service
            xtream = panel_manager.get_panel_for_service(service_id)
            result = xtream.search_user(username)
            
            if "error" in result:
                keyboard = [
                    [InlineKeyboardButton("🔄 Try Again", callback_data="my_accounts")],
                    [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
                ]
                reply_markup = InlineKeyboardMarkup(keyboard)
                
                await query.edit_message_text(
                    f"❌ <b>Error searching account</b>\n\n{result['error']}\n\nPlease try again or contact support.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return
            
            if result.get('found'):
                user_data = result['user_data']
                
                username_clean = clean_html_content(user_data['username'])
                password_clean = clean_html_content(user_data['password'])
                max_connections = clean_html_content(user_data['max_connections'])
                expiry_date = clean_html_content(user_data['expiration'])
                
                # Dynamic service emoji based on service number
                service_emoji = "📱"  # Default
                for service_num in range(1, 6):
                    if service['id'] == f"service{service_num}":
                        emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                        service_emoji = emoji_map.get(service_num, "📱")
                        break
                
                info_text = f"📋 <b>Account Information</b>\n\n"
                info_text += f"{service_emoji} <b>Service:</b> {service_name}\n"
                info_text += f"👤 <b>Username:</b> <code>{username_clean}</code>\n"
                info_text += f"🔑 <b>Password:</b> <code>{password_clean}</code>\n"
                info_text += f"🔗 <b>Max Connections:</b> {max_connections}\n"
                info_text += f"📅 <b>Expiry Date:</b> {expiry_date}"
                
                keyboard = []
                
                # ADD COPY BUTTONS HERE
                keyboard.append([
                    InlineKeyboardButton("📋 Copy Username", callback_data=f"copy_user_{username_clean}"),
                    InlineKeyboardButton("📋 Copy Password", callback_data=f"copy_pass_{username_clean}")
                ])
                
                # Check if renewal is allowed using centralized function
                can_renew, renewal_message = check_renewal_eligibility(username_clean, service_name, panel_manager)
                
                if can_renew:
                    keyboard.append([InlineKeyboardButton("🔄 Renew This Account", callback_data=f"renew_user_{username_clean}")])
                    info_text += f"\n\n🔄 <b>Renewal:</b> ✅ {renewal_message}"
                else:
                    info_text += f"\n\n⏰ <b>Renewal:</b> ❌ {renewal_message}"
                
                keyboard.extend([
                    [InlineKeyboardButton("🔄 Refresh Info", callback_data=f"check_account_{username}")],
                    [InlineKeyboardButton("🔗 Unlink Account", callback_data=f"unlink_account_{username_clean}")],
                    [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")]
                ])
                reply_markup = InlineKeyboardMarkup(keyboard)
                
                await query.edit_message_text(info_text, reply_markup=reply_markup, parse_mode='HTML')
            else:
                keyboard = [
                    [InlineKeyboardButton("🗑️ Remove Dead Account", callback_data=f"remove_dead_{username}")],
                    [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                    [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
                ]
                reply_markup = InlineKeyboardMarkup(keyboard)
                
                await query.edit_message_text(
                    f"❌ <b>Account Not Found</b>\n\nUsername '{username}' was not found in the {service_name} panel.\n\nThis account may have been deleted or expired. You can remove it from your linked accounts.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
    
    elif data.startswith("unlink_account_"):
        username = data.replace("unlink_account_", "")
        
        if db.unlink_account(user_id, username):
            keyboard = [
                [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                f"✅ <b>Account Unlinked</b>\n\nAccount '{username}' has been unlinked from your Telegram account.\n\nYou can link it again anytime if needed.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            await query.edit_message_text("❌ <b>Error</b>\n\nCould not unlink account. Please try again.")
    
    elif data.startswith("remove_dead_"):
        username = data.replace("remove_dead_", "")
        
        if db.remove_dead_account(user_id, username):
            keyboard = [
                [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                f"🗑️ <b>Dead Account Removed</b>\n\nAccount '{username}' has been removed from your linked accounts.\n\nThis was likely an expired or deleted account.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            await query.edit_message_text("❌ <b>Error</b>\n\nCould not remove account. Please try again.")
    
    elif data.startswith("copy_user_") or data.startswith("copy_pass_"):
        # Extract what we're copying and the username
        is_username = data.startswith("copy_user_")
        field = "Username" if is_username else "Password"
        username = data.replace("copy_user_", "").replace("copy_pass_", "")
        
        # Check if user can access this account
        if not db.can_access_account(user_id, username):
            try:
                await query.answer("❌ Access denied", show_alert=True)
            except Exception:
                pass  # Query may have already been answered
            return
        
        # Get the service for this account
        service_name = db.get_account_service(user_id, username)
        service = package_manager.get_service_by_name(service_name)
        
        if not service:
            try:
                await query.answer("❌ Service not found", show_alert=True)
            except Exception:
                pass  # Query may have already been answered
            return
        
        service_id = service['id']
        
        # Get panel and search for account
        panel = panel_manager.get_panel_for_service(service_id)
        if not panel:
            try:
                await query.answer("❌ Panel not available", show_alert=True)
            except Exception:
                pass  # Query may have already been answered
            return
        
        result = panel.search_user(username)
        
        if "error" in result or not result.get('found'):
            try:
                await query.answer("❌ Account not found", show_alert=True)
            except Exception:
                pass  # Query may have already been answered
            return
        
        user_data = result['user_data']
        value_to_copy = clean_html_content(user_data['username']) if is_username else clean_html_content(user_data['password'])
        
        # Show alert that it's ready to copy
        try:
            await query.answer(f"{field} ready to copy! Tap and hold the text below.", show_alert=True)
        except Exception:
            pass  # Query may have already been answered
        
        # Create a focused view for copying
        copy_text = f"📋 <b>{field} for {service_name}</b>\n\n"
        copy_text += f"<b>Account:</b> {username}\n"
        copy_text += f"<b>{field}:</b> <code>{value_to_copy}</code>\n\n"
        copy_text += f"👆 <i>Tap and hold the text above to copy</i>"
        
        keyboard = [
            [InlineKeyboardButton("📱 Back to Account", callback_data=f"check_account_{username}")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            copy_text,
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data == "link_account":
        db.clear_user_state(user_id)

        # Show service selection first
        services = package_manager.get_services()

        if not services:
            await query.edit_message_text(
                "❌ <b>No Services Available</b>\n\nNo services are configured.",
                parse_mode='HTML'
            )
            return

        keyboard = []
        for service in services:
            if service['id'] == "add_funds":
                continue  # Skip Add Funds from the linkable service list
            callback_data = f"link_select_service_{service['id']}"
            keyboard.append([InlineKeyboardButton(f"🔘 {service['name']}", callback_data=callback_data)])

        keyboard.append([InlineKeyboardButton("❌ Cancel", callback_data="back_to_menu")])
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(
            "🔗 <b>Link Account - Select Service</b>\n\n"
            "Which service is this account for?",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data.startswith("link_select_service_"):
        service_id = data.replace("link_select_service_", "")
        service = package_manager.get_service_by_id(service_id)
        
        if not service:
            await query.edit_message_text("❌ Service not found", parse_mode='HTML')
            return
        
        db.set_user_state(user_id, "link_account_username", service['name'])
        
        keyboard = [[InlineKeyboardButton("❌ Cancel", callback_data="cancel_link_account")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            f"🔗 <b>Link Account - {service['name']}</b>\n\n"
            f"Please enter the username of your {service['name']} account:\n\n"
            "<b>Requirements:</b>\n"
            "• Must be your own account\n"
            "• Account must exist in our panels\n"
            "• You'll need to verify ownership\n\n"
            "Please type the username:",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
    
    elif data == "cancel_link_account":
        db.clear_user_state(user_id)
        # Return user to main menu (or wherever you want)
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        welcome_text = f"""🎬 <b>User Login Portal</b>

    Welcome back {query.from_user.first_name}! 

    You've cancelled account linking. 
    Choose an option below:"""
        reply_markup = get_main_menu_keyboard(user_id)
        await query.edit_message_text(welcome_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "renew_account":
        linked_accounts = db.get_linked_accounts(user_id)
        
        if not linked_accounts:
            keyboard = [
                [InlineKeyboardButton("🔗 Link Account", callback_data="link_account")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                "🔄 <b>Renew Account</b>\n\nYou need to link an account first before you can renew it.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            # Set state to indicate this is from main menu
            db.set_user_state(user_id, "renewal_from_main_menu", "")
            
            # Show loading message first
            await query.edit_message_text(
                "🔄 <b>Checking Account Renewal Eligibility...</b>\n\n"
                f"📊 Checking {len(linked_accounts)} linked account(s)...\n"
                f"⏳ This may take a few seconds...\n\n"
                f"🔍 Verifying account status and expiry dates...",
                parse_mode='HTML'
            )
            
            # Add a small delay to show the loading message
            import asyncio
            await asyncio.sleep(1)
            
            keyboard = []
            renewal_available_accounts = []
            accounts_checked = 0
            total_accounts = len(linked_accounts)
            
            for username, service_name in linked_accounts:
                accounts_checked += 1
                
                # Update progress message for longer checks
                if total_accounts > 3 and accounts_checked % 2 == 0:  # Update every 2 accounts if more than 3
                    progress_text = f"🔄 <b>Checking Account Renewal Eligibility...</b>\n\n"
                    progress_text += f"📊 Progress: {accounts_checked}/{total_accounts} accounts checked\n"
                    progress_text += f"⏳ Checking: {username} ({service_name})...\n\n"
                    progress_text += f"🔍 Verifying account status and expiry dates..."
                    
                    try:
                        await query.edit_message_text(progress_text, parse_mode='HTML')
                        await asyncio.sleep(0.5)  # Brief pause between updates
                    except:
                        pass  # Ignore edit conflicts
                
                # Check if account is eligible for renewal using centralized function
                can_renew, renewal_message = check_renewal_eligibility(username, service_name, panel_manager)

                if can_renew:
                    # Dynamic service emoji based on service
                    service_emoji = "📱"  # Default
                    service = package_manager.get_service_by_name(service_name)
                    if service:
                        for service_num in range(1, 6):
                            if service['id'] == f"service{service_num}":
                                emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                                service_emoji = emoji_map.get(service_num, "📱")
                                break
                    
                    keyboard.append([InlineKeyboardButton(f"🔄 {service_emoji} {username} ({service_name})", callback_data=f"renew_user_{username}")])
                    renewal_available_accounts.append((username, service_name, renewal_message))
            
            # Show final results
            if not renewal_available_accounts:
                keyboard = [
                    [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                    [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
                ]
                reply_markup = InlineKeyboardMarkup(keyboard)
                
                await query.edit_message_text(
                    "🔄 <b>Renewal Check Complete</b>\n\n"
                    "⏰ No accounts are currently eligible for renewal.\n\n"
                    "<b>📋 Requirements for Renewal:</b>\n"
                    "• Account must expire within 10 days\n"
                    "• Account must exist in the panel\n"
                    "• Account must be active\n\n"
                    f"✅ Checked {total_accounts} account(s) successfully.\n\n"
                    "Check your account details in 'My Accounts' to see when renewal will be available.",
                    reply_markup=reply_markup,
                    parse_mode='HTML'
                )
                return
            
            # Build the keyboard with individual renewal buttons
            keyboard = []
            for username, service_name, renewal_message in renewal_available_accounts:
                # Dynamic service emoji based on service
                service_emoji = "📱"  # Default
                service = package_manager.get_service_by_name(service_name)
                if service:
                    for service_num in range(1, 6):
                        if service['id'] == f"service{service_num}":
                            emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                            service_emoji = emoji_map.get(service_num, "📱")
                            break
                
                keyboard.append([InlineKeyboardButton(f"🔄 {service_emoji} {username} ({service_name})", callback_data=f"renew_user_{username}")])
            
            # Add bulk renewal button if multiple accounts are eligible BEFORE adding home button
            if len(renewal_available_accounts) > 1:
                keyboard.insert(0, [InlineKeyboardButton(f"🔄 Renew All ({len(renewal_available_accounts)} accounts)", callback_data="renew_all")])
            
            # Add home button at the end
            keyboard.append([InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")])
            
            # Create the renewal text
            renewal_text = "🔄 <b>Renewal Check Complete</b>\n\n"
            renewal_text += f"✅ Found {len(renewal_available_accounts)} account(s) eligible for renewal:\n\n"
            
            for username, service_name, renewal_message in renewal_available_accounts:
                service_emoji = "📱"
                service = package_manager.get_service_by_name(service_name)
                if service:
                    for service_num in range(1, 6):
                        if service['id'] == f"service{service_num}":
                            emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                            service_emoji = emoji_map.get(service_num, "📱")
                            break
                
                renewal_text += f"• {service_emoji} <b>{username}</b> ({service_name})\n"
                renewal_text += f"  └ {renewal_message}\n\n"
            
            if len(renewal_available_accounts) < len(linked_accounts):
                accounts_not_eligible = len(linked_accounts) - len(renewal_available_accounts)
                renewal_text += f"ℹ️ <i>{accounts_not_eligible} account(s) not yet eligible (must be within 10 days of expiry)</i>\n\n"
            
            renewal_text += "Select an account to renew:"
            
            # NOW create reply_markup AFTER all buttons are added
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                renewal_text,
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
    
    elif data == "renew_all":
        # Get all eligible accounts
        linked_accounts = db.get_linked_accounts(user_id)
        eligible_accounts = []
        total_price = 0.0
        
        # Check each account and calculate total
        for username, service_name in linked_accounts:
            can_renew, _ = check_renewal_eligibility(username, service_name, panel_manager)
            if can_renew:
                # Get service and default pricing
                service = package_manager.get_service_by_name(service_name)
                if service and service.get('billing_cycles'):
                    # Use first cycle (usually 30 days) as default
                    default_cycle = service['billing_cycles'][0]
                    price = default_cycle.get('price', 0)
                    total_price += price
                    
                    # Get the first package's panel_package_id
                    if service.get('order_type') == 'matrix':
                        # For matrix services, get first connection's panel package ID
                        if 'connections' in default_cycle and default_cycle['connections']:
                            panel_package_id = default_cycle['connections'][0].get('panel_package_no_xxx', '')
                        else:
                            panel_package_id = ''
                    else:
                        # For direct cycle services
                        panel_package_id = default_cycle.get('panel_package_id', '')
                    
                    eligible_accounts.append({
                        'username': username,
                        'service_name': service_name,
                        'service_id': service['id'],
                        'price': price,
                        'cycle': default_cycle,
                        'panel_package_id': panel_package_id
                    })
        
        if not eligible_accounts:
            await query.edit_message_text("❌ No eligible accounts found.", parse_mode='HTML')
            return
        
        # Show confirmation
        confirmation_text = f"🔄 <b>Bulk Renewal Confirmation</b>\n\n"
        confirmation_text += f"<b>Renewing {len(eligible_accounts)} accounts:</b>\n\n"
        
        for acc in eligible_accounts:
            confirmation_text += f"• {acc['username']} ({acc['service_name']}) - ${acc['price']:.2f}\n"
        
        confirmation_text += f"\n💰 <b>Total: ${total_price:.2f}</b>\n\n"
        confirmation_text += "✅ All accounts will be renewed with ONE payment!"
        
        # Store bulk renewal data in user state
        import json
        db.set_user_state(user_id, "bulk_renewal", json.dumps(eligible_accounts))
        
        keyboard = [
            [InlineKeyboardButton("✅ Confirm Bulk Renewal", callback_data="confirm_bulk_renewal")],
            [InlineKeyboardButton("❌ Cancel", callback_data="renew_account")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(confirmation_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "confirm_bulk_renewal":
        # Get stored bulk renewal data
        state, data = db.get_user_state(user_id)
        if state != "bulk_renewal":
            await query.edit_message_text("❌ Session expired. Please try again.", parse_mode='HTML')
            return
        
        import json
        eligible_accounts = json.loads(data)
        
        # Calculate total
        total_price = sum(acc['price'] for acc in eligible_accounts)
        
        # Create a SINGLE bulk order
        order_id, bot_payment_ref = db.add_order_with_payment(
            telegram_user_id=user_id,
            telegram_username=query.from_user.username,
            order_type="bulk_renewal",
            service_id="multiple",
            service_name="Multiple Services",
            package_id="bulk",
            package_name=f"Bulk Renewal ({len(eligible_accounts)} accounts)",
            billing_cycle="30days",
            panel_package_id="",
            base_price=total_price,
            xxx_addon=False,
            setup_fee=0,
            total_price=total_price,
            cashapp_tag=os.getenv("SERVICE1_CASHAPP_TAG", "$Service918"),
            username="",  # We'll store account list in details
            password="",
            details=json.dumps(eligible_accounts)  # Store all account info
        )
        
        # Clear state
        db.clear_user_state(user_id)
        
        # Show payment instructions
        payment_text = f"💳 <b>Bulk Renewal Payment</b>\n\n"
        payment_text += f"<b>Order ID:</b> #{order_id}\n"
        payment_text += f"<b>Payment Reference:</b> <code>{bot_payment_ref}</code>\n"
        payment_text += f"<b>Accounts:</b> {len(eligible_accounts)} renewals\n"
        payment_text += f"<b>Total Amount:</b> ${total_price:.2f}\n\n"
        
        payment_text += f"📱 <b>ONE Payment for ALL Renewals!</b>\n\n"
        
        payment_text += f"<b>Accounts being renewed:</b>\n"
        for acc in eligible_accounts:
            payment_text += f"• {acc['username']} ({acc['service_name']})\n"
        
        payment_text += f"\n💳 Send ${total_price:.2f} with reference: <code>{bot_payment_ref}</code>"
        
        # Create Cash App deeplink
        cashapp_tag = os.getenv("SERVICE1_CASHAPP_TAG", "$Service918").replace('$', '')
        cashapp_url = f"https://cash.app/{cashapp_tag}/{total_price:.2f}"
        
        keyboard = [
            [InlineKeyboardButton("💳 Pay with Cash App", url=cashapp_url)],
            [InlineKeyboardButton("✅ I've Sent Payment", callback_data=f"payment_sent_{order_id}")],
            [InlineKeyboardButton("📋 Check Payment Status", callback_data=f"check_payment_{order_id}")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(payment_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data.startswith("renew_user_"):
        username = data.replace("renew_user_", "")
        
        # Check if user can access this account
        if not db.can_access_account(user_id, username):
            await query.edit_message_text("❌ <b>Access Denied</b>\n\nYou don't have permission to renew this account.", parse_mode='HTML')
            return
        
        # Process the renewal
        await process_direct_renewal(context, chat_id, message_id, user_id, username)

    elif data == "my_orders":
        orders = db.get_user_orders(user_id)
        
        if not orders:
            keyboard = [
                [InlineKeyboardButton("🛒 Place New Order", callback_data="new_order")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                "📋 <b>My Orders</b>\n\nYou don't have any orders yet.\n\nPlace your first order to get started!",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            order_text = "📋 <b>My Orders</b>\n\n"
            
            for order in orders[:10]:  # Show last 10 orders
                order_id, order_type, service_name, package_name, billing_cycle, total_price, status, payment_status, created_at, xxx_addon, username, password = order
                
                # Format date
                try:
                    created_date = datetime.fromisoformat(created_at).strftime("%Y-%m-%d")
                except:
                    created_date = created_at[:10] if created_at else "Unknown"
                
                # Status emoji
                if status == "completed":
                    status_emoji = "✅"
                elif status == "cancelled":
                    status_emoji = "❌"
                elif payment_status == "verified":
                    status_emoji = "⏳"
                else:
                    status_emoji = "⏰"
                
                order_text += f"{status_emoji} <b>Order #{order_id}</b>\n"
                order_text += f"📦 {service_name} - {package_name}\n"
                order_text += f"💰 ${total_price:.2f} • {created_date}\n"
                order_text += f"🔄 Status: {status.title()}\n\n"
            
            keyboard = [
                [InlineKeyboardButton("🛒 New Order", callback_data="new_order")],
                [InlineKeyboardButton("🔄 Renew Account", callback_data="renew_account")],
                [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(order_text, reply_markup=reply_markup, parse_mode='HTML')
    
    elif data == "help":
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        help_text = f"""ℹ️ <b>{bot_name} - Help & Support</b>

<b>🔗 Account Linking:</b>
• Link your existing accounts to manage them securely
• Only you can access your linked accounts
• Required for renewals and account checking

<b>🔄 Renewals:</b>
• Available within 10 days of expiry
• No setup fees for renewals
• Automatic account extension

<b>🛒 New Orders:</b>
• Choose your service and package
• Secure Cash App payments
• Instant account creation after payment

<b>💰 Pricing:</b>
• Setup fees apply to new accounts only
• XXX content available for supported services
• Multiple billing cycles available

<b>🔒 Security:</b>
• All accounts are verified before linking
• Secure payment processing
• Account information is encrypted

<b>💬 Need Help?</b>
If you have questions or issues:
• Check your account status first
• Contact our support team
• Include your order number if applicable

<b>📞 Support Contact:</b>
Contact support through this bot or our official channels."""
        
        keyboard = [
            [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
            [InlineKeyboardButton("🔗 Link Account", callback_data="link_account")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(help_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "about_page":
        try:
            import json as json_module  # Import locally to avoid any conflicts
            
            # Check if file exists first
            if not os.path.exists("about.json"):
                message = "⚠️ About file not found. Please ensure about.json is in the same directory as the bot."
                print("ERROR: about.json file not found!")
            else:
                with open("about.json", "r", encoding="utf-8") as f:
                    about_content = json_module.load(f)  # Use different variable name
                
                message = "ℹ️ <b>About Our Services</b>\n\n"
                if "info" in about_content:
                    message += f"{about_content['info']}\n\n"
                if "services" in about_content:
                    message += "<b>Services:</b>\n"
                    for svc in about_content["services"]:
                        message += f"• <b>{svc['name']}</b>: {svc.get('description','')}\n"
                        if svc.get("setup"):
                            message += f"  <i>Setup:</i> {svc['setup']}\n"
                        message += "\n"
                if "instructions" in about_content:
                    message += f"<b>Instructions:</b> {about_content['instructions']}\n\n"
                if "support_link" in about_content:
                    message += f"💬 <a href=\"{about_content['support_link']}\">Support Group</a>"
        except json_module.JSONDecodeError as e:
            message = f"⚠️ Error parsing about.json: Invalid JSON format at line {e.lineno}"
            print(f"JSON ERROR: {e}")
        except Exception as e:
            message = f"⚠️ Error loading about information: {str(e)}"
            print(f"ERROR loading about.json: {e}")

        keyboard = [
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(
            message,
            reply_markup=reply_markup,
            parse_mode="HTML",
            disable_web_page_preview=True
        )

    elif data == "offers":
        try:
            import json as json_module  # Import locally to avoid any conflicts
            
            # Check if file exists first
            if not os.path.exists("offers.json"):
                message = "⚠️ Offers file not found. Please ensure offers.json is in the same directory as the bot."
                print("ERROR: about.json file not found!")
            else:
                with open("offers.json", "r", encoding="utf-8") as f:
                    offers_content = json_module.load(f)  # Use different variable name
                
                message = "🔥 <b>Special Offers</b>\n\n"
                if "info" in offers_content:
                    message += f"{offers_content['info']}\n\n"
                if "offers" in offers_content:
                    message += "<b>Offers:</b>\n"
                    for svc in offers_content["offers"]:
                        message += f"• <b>{svc['service']}</b>: {svc.get('deal','')}\n"
                        if svc.get("req") and svc["req"].strip():
                            message += f"  <i>Requirement:</i> <u>{svc['req']}</u>\n"
                        if svc.get("code"):
                            message += f"  <i>Promo Code:</i> <b>{svc['code']}</b>\n"
                        message += "\n"
                if "instructions" in offers_content:
                    message += f"<b>Instructions:</b> {offers_content['instructions']}\n\n"
        except json_module.JSONDecodeError as e:
            message = f"⚠️ Error parsing offers.json: Invalid JSON format at line {e.lineno}"
            print(f"JSON ERROR: {e}")
        except Exception as e:
            message = f"⚠️ Error loading about information: {str(e)}"
            print(f"ERROR loading offers.json: {e}")

        keyboard = [
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(
            message,
            reply_markup=reply_markup,
            parse_mode="HTML",
            disable_web_page_preview=True
        )

    elif data == "admin_panel":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ <b>Access Denied</b>\n\nYou don't have admin privileges.", parse_mode='HTML')
            return
        
        # Get pending orders and link requests counts
        pending_orders = db.get_pending_orders()
        pending_links = db.get_pending_link_requests()
        
        admin_text = f"👨‍💼 <b>Admin Panel</b>\n\n"
        admin_text += f"📊 <b>System Status:</b>\n"
        admin_text += f"• Pending Orders: {len(pending_orders)}\n"
        admin_text += f"• Pending Links: <b>{len(pending_links)}</b>\n\n"
        admin_text += f"Select an option:"
        
        new_tickets = get_new_ticket_count()
        tickets_label = f"🎫 Tickets ({new_tickets})" if new_tickets > 0 else "🎫 Tickets"

        keyboard = [
            [
                InlineKeyboardButton(f"📋 Pending Orders ({len(pending_orders)})", callback_data="admin_orders"),
                InlineKeyboardButton("➕ Add Funds to User", callback_data="admin_add_funds_user")
            ],
            [
                InlineKeyboardButton("➖ Remove Funds from User", callback_data="admin_remove_funds_user"),
                InlineKeyboardButton("👥 User List (DM)", callback_data="admin_userlist")
            ],
            [
                InlineKeyboardButton(
                    f"🔗 Link Requests ({len(pending_links)})" + (" 🔴" if len(pending_links) > 0 else ""),
                    callback_data="admin_links"
                ),
                InlineKeyboardButton("🧹 Prune Dead Accounts", callback_data="admin_prune")
            ],
            [
                InlineKeyboardButton("📊 System Stats", callback_data="admin_stats"),
                InlineKeyboardButton("🧨 Reset Stats", callback_data="admin_reset_stats")
            ],
            [InlineKeyboardButton(tickets_label, callback_data="admin_ticket_menu")],
            [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(admin_text, reply_markup=reply_markup, parse_mode='HTML')
        
    elif data == "admin_orders":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        pending_orders = db.get_pending_orders()

        if not pending_orders:
            keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
            reply_markup = InlineKeyboardMarkup(keyboard)

            await query.edit_message_text(
                "📋 <b>Pending Orders</b>\n\nNo pending orders found.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            orders_text = "📋 <b>Pending Orders</b>\n\n"
            keyboard = []
            for order in pending_orders[:10]:  # Show first 10
                order_id, telegram_user_id, order_type, service_name, package_name, billing_cycle, total_price, username, details, created_at, telegram_username = order

                try:
                    created_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
                except:
                    created_date = created_at[:16] if created_at else "Unknown"

                orders_text += f"🆔 <b>#{order_id}</b> - ${total_price:.2f}\n"
                orders_text += f"👤 @{telegram_username} ({telegram_user_id})\n"
                orders_text += f"📦 {service_name} - {package_name}\n"
                orders_text += f"📅 {created_date} • {order_type}\n\n"

                # Add cancel button for each order
                keyboard.append([InlineKeyboardButton(f"✅ Confirm #{order_id}", callback_data=f"admin_confirm_order_{order_id}"), InlineKeyboardButton(f"❌ Cancel #{order_id}", callback_data=f"admin_cancel_order_{order_id}")])

            # Only add these ONCE, after the loop
            if pending_orders:
                keyboard.append([InlineKeyboardButton("❌❌ Cancel All Pending Orders", callback_data="admin_cancel_all_orders")])
            keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
            reply_markup = InlineKeyboardMarkup(keyboard)

            await query.edit_message_text(orders_text, reply_markup=reply_markup, parse_mode='HTML')
    
    # ----- this is the admin userlist block -----
    elif data.startswith("admin_userlist"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        # Parse for page or search
        max_per_page = 10
        page = 0
        search_term = None
        if "|" in data:
            parts = data.split("|")
            if len(parts) == 2 and parts[1].isdigit():
                page = int(parts[1])
            elif len(parts) == 3 and parts[1] == "search":
                search_term = parts[2].strip().lower()
                page = 0
            elif len(parts) == 4 and parts[1] == "search" and parts[3].isdigit():
                search_term = parts[2].strip().lower()
                page = int(parts[3])

        # Fetch all users (recent first)
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        if search_term:
            cursor.execute(
                "SELECT user_id, telegram_username, is_admin, created_at FROM users WHERE telegram_username LIKE ? OR user_id LIKE ? ORDER BY created_at DESC",
                (f"%{search_term}%", f"%{search_term}%")
            )
        else:
            cursor.execute(
                "SELECT user_id, telegram_username, is_admin, created_at FROM users ORDER BY created_at DESC"
            )
        all_users = cursor.fetchall()
        conn.close()

        user_count = len(all_users)
        start = page * max_per_page
        end = start + max_per_page
        page_users = all_users[start:end]

        # Message: clickable username/ID links
        userlist_text = "👥 <b>User List</b>\n\n"
        if search_term:
            userlist_text += f"🔍 <i>Search results for:</i> <b>{search_term}</b>\n\n"

        if not page_users:
            userlist_text += "<i>No users found.</i>\n"
        else:
            for u_id, t_username, is_admin, created_at in page_users:
                label = f"{'👑' if is_admin else ''} @{t_username or '(no_username)'} ({u_id})"
                userlist_text += f"• <a href='tg://user?id={u_id}'>{label}</a>\n"

        # Pagination controls
        keyboard = []
        nav_row = []
        if start > 0:
            if search_term:
                nav_row.append(InlineKeyboardButton("⬅️ Prev", callback_data=f"admin_userlist|search|{search_term}|{page-1}"))
            else:
                nav_row.append(InlineKeyboardButton("⬅️ Prev", callback_data=f"admin_userlist|{page-1}"))
        if end < user_count:
            if search_term:
                nav_row.append(InlineKeyboardButton("➡️ Next", callback_data=f"admin_userlist|search|{search_term}|{page+1}"))
            else:
                nav_row.append(InlineKeyboardButton("➡️ Next", callback_data=f"admin_userlist|{page+1}"))
        if nav_row:
            keyboard.append(nav_row)

        # Search button
        keyboard.append([InlineKeyboardButton("🔎 Search Users", callback_data="admin_userlist_search")])

        # Back button
        keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(
            userlist_text,
            reply_markup=reply_markup,
            parse_mode='HTML',
            disable_web_page_preview=True
        )

# ----- this ends the admin userlist block -----

# ----- this is the admin userlist search prompt block -----
    elif data == "admin_userlist_search":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return
        db.set_user_state(user_id, "admin_userlist_search", "")
        keyboard = [[InlineKeyboardButton("« Back to User List", callback_data="admin_userlist")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.message.reply_text(
            "🔎 <b>Search Users</b>\n\n"
            "Enter a Telegram username or user ID to search for:",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
# ----- this ends the admin userlist search prompt block -----

    elif data == "admin_add_funds_user":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        # Ask admin for user ID or username
        db.set_user_state(user_id, "admin_add_funds_target", "")
        keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "➕ <b>Add Funds to User</b>\n\n"
            "Please enter the user's Telegram ID or @username:",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data == "admin_remove_funds_user":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        db.set_user_state(user_id, "admin_remove_funds_target", "")
        keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "➖ <b>Remove Funds from User</b>\n\n"
            "Please enter the user's Telegram ID or @username:",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data == "admin_cancel_all_orders":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        # Confirm action first
        keyboard = [
            [InlineKeyboardButton("⚠️ Confirm Cancel All", callback_data="admin_confirm_cancel_all_orders")],
            [InlineKeyboardButton("« Back to Pending Orders", callback_data="admin_orders")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "⚠️ <b>Are you sure you want to cancel ALL pending orders?</b>\n\n"
            "This action cannot be undone.",
            reply_markup=reply_markup,
            parse_mode="HTML"
        )
    elif data == "admin_confirm_cancel_all_orders":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        # Cancel all pending orders
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE orders SET status='cancelled', notes='Cancelled by admin (bulk)', updated_at=CURRENT_TIMESTAMP "
            "WHERE status IN ('pending_payment', 'pending')"
        )
        affected = cursor.rowcount
        conn.commit()
        conn.close()

        await query.edit_message_text(
            f"❌ <b>All pending orders cancelled.</b>\n\n"
            f"Total cancelled: <b>{affected}</b>",
            parse_mode='HTML'
        )

        # Optionally, refresh the pending orders list (uncomment if desired)
        await show_admin_orders(query, context)

    elif data == "admin_links":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return
        
        pending_links = db.get_pending_link_requests()
        
        if not pending_links:
            keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await query.edit_message_text(
                "🔗 <b>Link Requests</b>\n\nNo pending link requests found.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            links_text = "🔗 <b>Pending Link Requests</b>\n\n"
            keyboard = []

            for link_request in pending_links:  # Show first 5
                link_id, telegram_user_id, panel_username, verification_code, created_at, telegram_username = link_request

                try:
                    created_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
                except:
                    created_date = created_at[:16] if created_at else "Unknown"

                links_text += f"🆔 <b>#{link_id}</b>\n"
                if telegram_username:
                    links_text += f"👤 <a href='tg://user?id={telegram_user_id}'>@{telegram_username}</a> (<code>{telegram_user_id}</code>)\n"
                else:
                    links_text += f"👤 <code>{telegram_user_id}</code>\n"
                links_text += f"🔑 Username: <b>{panel_username}</b>\n"

                # Get service name from the link request
                conn = sqlite3.connect(db.db_path)
                cursor = conn.cursor()
                cursor.execute("SELECT service_name FROM account_links WHERE id = ?", (link_id,))
                service_result = cursor.fetchone()
                conn.close()
                service_name = service_result[0] if service_result else "Unknown"

                links_text += f"🔘 Service: <b>{service_name}</b>\n"
                links_text += f"📝 Code: <code>{verification_code}</code>\n"
                links_text += f"📅 {created_date}\n\n"

                keyboard.append([InlineKeyboardButton(f"✅ Approve #{link_id}", callback_data=f"approve_link_{link_id}")])
                keyboard.append([InlineKeyboardButton(f"❌ Reject #{link_id}", callback_data=f"reject_link_{link_id}")])

            # <<< OUTSIDE the for loop but INSIDE the else block >>>
            keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
            reply_markup = InlineKeyboardMarkup(keyboard)
            await query.edit_message_text(links_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data.startswith("approve_link_"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        link_id = int(data.split("_")[-1])
        success, msg = db.approve_account_link(link_id, user_id)

        if success:
            await query.edit_message_text(
                f"✅ <b>Link Approved</b>\n\nLink request #{link_id} has been approved.",
                parse_mode='HTML'
            )

            # Notify the user of approval **ONLY INSIDE THIS BLOCK**
            try:
                conn = sqlite3.connect(db.db_path)
                cursor = conn.cursor()
                cursor.execute("SELECT telegram_user_id, panel_username FROM account_links WHERE id = ?", (link_id,))
                row = cursor.fetchone()
                conn.close()
                if row:
                    approved_user_id, panel_username = row
                    try:
                        await context.bot.send_message(
                            chat_id=approved_user_id,
                            text=(
                                f"✅ <b>Your account link request for <code>{panel_username}</code> was approved!</b>\n\n"
                                "Your Telegram is now linked to your panel account. If you have any questions, contact support."
                            ),
                            parse_mode="HTML"
                        )
                    except Exception as e:
                        print(f"[WARN] Could not notify user {approved_user_id}: {e}")
            except Exception as e:
                print(f"[WARN] Error in approval notification: {e}")

        else:
            await query.edit_message_text(
                f"❌ <b>Approval Blocked</b>\n\n{msg}",
                parse_mode='HTML'
            )

        # Refresh admin links view after 2 seconds
        await asyncio.sleep(2)
        pending_links = db.get_pending_link_requests()
        if not pending_links:
            keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await query.edit_message_text(
                "🔗 <b>Link Requests</b>\n\nNo pending link requests found.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            links_text = "🔗 <b>Pending Link Requests</b>\n\n"
            keyboard = []
            for link_request in pending_links:
                l_id, telegram_user_id, panel_username, verification_code, created_at, telegram_username = link_request
                try:
                    created_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
                except:
                    created_date = created_at[:16] if created_at else "Unknown"
                if telegram_username:
                    links_text += f"🆔 <b>#{l_id}</b>\n"
                    links_text += f"👤 <a href='tg://user?id={telegram_user_id}'>@{telegram_username}</a> (<code>{telegram_user_id}</code>)\n"
                else:
                    links_text += f"🆔 <b>#{l_id}</b>\n"
                    links_text += f"👤 <code>{telegram_user_id}</code>\n"
                links_text += f"🔑 Username: <b>{panel_username}</b>\n"
                links_text += f"📝 Code: <code>{verification_code}</code>\n"
                links_text += f"📅 {created_date}\n\n"
                keyboard.append([InlineKeyboardButton(f"✅ Approve #{l_id}", callback_data=f"approve_link_{l_id}")])
                keyboard.append([InlineKeyboardButton(f"❌ Reject #{l_id}", callback_data=f"reject_link_{l_id}")])
            keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
            reply_markup = InlineKeyboardMarkup(keyboard)
            await query.edit_message_text(
                links_text,
                reply_markup=reply_markup,
                parse_mode='HTML',
                disable_web_page_preview=True
            )
    
    elif data.startswith("reject_link_"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        link_id = int(data.split("_")[-1])

        # Show confirmation before rejecting
        keyboard = [
            [InlineKeyboardButton("⚠️ Confirm Reject", callback_data=f"confirm_reject_link_{link_id}")],
            [InlineKeyboardButton("« Back to Admin", callback_data="admin_links")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(
            f"⚠️ <b>Are you sure you want to reject link request #{link_id}?</b>\n\n"
            "This cannot be undone.",
            reply_markup=reply_markup,
            parse_mode="HTML"
        )

    elif data.startswith("admin_confirm_order_"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        order_id = int(data.split("_")[-1])
        order_data = db.get_order_by_id(order_id)
        if not order_data:
            await query.edit_message_text("❌ Order not found.", parse_mode='HTML')
            return

        # Extract relevant info
        order_type = order_data['order_type']
        service_id = order_data['service_id']
        service = package_manager.get_service_by_id(service_id)
        panel_package_id = order_data['panel_package_id']
        user_telegram_id = order_data['telegram_user_id']
        package_name = order_data['package_name']
        username = order_data['username']
        password = order_data['password']
        telegram_username = order_data['telegram_username']
        total_price = order_data['total_price']

        result_msg = ""
        credentials_message = ""
        reply_markup = None

        # Mark as fulfilled by admin (override status)
        db.update_order_status(order_id, "completed", "Manually fulfilled by admin (override)")

        # Process in panel as if payment was received
        if order_type == "renewal":
            panel = panel_manager.get_panel_for_service(service_id)
            if panel and username:
                renewal_result = panel.renew_user_account(
                    username=username,
                    package_id=panel_package_id,
                    duration_days=30,
                    notes=f"Manual admin renewal for order #{order_id}"
                )
                if renewal_result.get("success"):
                    result_msg = (
                        f"✅ <b>Your order #{order_id} has been manually fulfilled by support.</b>\n\n"
                        f"Your account <code>{username}</code> has been renewed!\n"
                        f"New expiry: {renewal_result.get('new_expiry', 'N/A')}\n\n"
                        f"If you have questions, please contact us."
                    )
                else:
                    result_msg = (
                        f"❌ <b>Your order #{order_id} could not be renewed in the panel.</b>\n\n"
                        f"Error: {renewal_result.get('error', 'Unknown error')}\n\n"
                        f"Please contact support."
                    )
                await context.bot.send_message(
                    chat_id=user_telegram_id,
                    text=result_msg,
                    parse_mode="HTML"
                )
            else:
                await context.bot.send_message(
                    chat_id=user_telegram_id,
                    text=(
                        f"✅ <b>Your order #{order_id} has been manually fulfilled by support.</b>\n\n"
                        f"If you have questions, please contact us."
                    ),
                    parse_mode="HTML"
                )
        else:
            # New order (create account in panel)
            panel = panel_manager.get_panel_for_service(service_id)
            if panel:
                create_result = panel.create_user_account(
                    username=username,
                    password=password,
                    package_id=panel_package_id,
                    duration_days=30,
                    max_connections=1,
                    notes=f"Manual admin creation for order #{order_id}"
                )
                if create_result.get('success'):
                    actual_username = create_result.get('username', username)
                    actual_password = create_result.get('password', password)
                    db.update_order_status(order_id, "completed", "Account created successfully (by admin)")
                    # AUTO-LINK to user's My Accounts (if not already linked)
                    try:
                        conn = sqlite3.connect(db.db_path)
                        cursor = conn.cursor()
                        cursor.execute("""
                            SELECT id FROM account_links 
                            WHERE telegram_user_id = ? AND panel_username = ?
                        """, (user_telegram_id, actual_username))
                        existing_link = cursor.fetchone()
                        if not existing_link:
                            cursor.execute("""
                                INSERT INTO account_links (telegram_user_id, panel_username, service_name, status, verification_code, linked_at, approved_by, created_at)
                                VALUES (?, ?, ?, 'approved', ?, CURRENT_TIMESTAMP, 0, CURRENT_TIMESTAMP)
                            """, (user_telegram_id, actual_username, service['name'], f"AUTO_ORDER_{order_id}"))
                            conn.commit()
                        conn.close()
                    except Exception as e:
                        print(f"[WARN] Failed to auto-link account {actual_username} for user {user_telegram_id}: {e}")
                    # Send credentials to user
                    credentials_message = f"""✅ <b>Order Completed!</b>

    <b>Order ID:</b> #{order_id}
    <b>Service:</b> {service['name']}
    <b>Package:</b> {package_name}

    🔑 <b>Your Login Details:</b>

    <b>Username:</b> <code>{actual_username}</code>
    <b>Password:</b> <code>{actual_password}</code>

    <b>Important:</b>
    • Save these credentials securely
    • Account has been automatically linked to your Telegram
    • You can view this account in "My Accounts"
    • Renewal available within 10 days of expiry

    Thank you for your order! 🎉"""

                    reply_markup = InlineKeyboardMarkup([
                        [
                            InlineKeyboardButton("📋 Copy Username", callback_data=f"copy_user_{actual_username}"),
                            InlineKeyboardButton("📋 Copy Password", callback_data=f"copy_pass_{actual_username}")
                        ],
                        [InlineKeyboardButton("📱 My Accounts", callback_data="my_accounts")],
                        [InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]
                    ])
                    await context.bot.send_message(
                        chat_id=user_telegram_id,
                        text=credentials_message,
                        reply_markup=reply_markup,
                        parse_mode="HTML"
                    )
                else:
                    await context.bot.send_message(
                        chat_id=user_telegram_id,
                        text=(
                            f"❌ <b>Your order #{order_id} could not be fulfilled in the panel.</b>\n\n"
                            f"Error: {create_result.get('error', 'Unknown error')}\n\n"
                            f"Please contact support."
                        ),
                        parse_mode="HTML"
                    )
            else:
                await context.bot.send_message(
                    chat_id=user_telegram_id,
                    text=(
                        f"✅ <b>Your order #{order_id} has been manually fulfilled by support.</b>\n\n"
                        f"If you have questions, please contact us."
                    ),
                    parse_mode="HTML"
                )

        await query.edit_message_text(
            f"✅ <b>Order #{order_id} has been confirmed and marked as fulfilled.</b>",
            parse_mode='HTML'
        )

        # Refresh the admin orders view
        await show_admin_orders(query, context)

    elif data.startswith("admin_cancel_order_"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        order_id = int(data.split("_")[-1])
        db.update_order_status(order_id, "cancelled", "Cancelled by admin")

        # Optionally notify the user
        order_data = db.get_order_by_id(order_id)
        if order_data:
            user_telegram_id = order_data['telegram_user_id']
            bot_payment_ref = order_data['notes']
            try:
                await context.bot.send_message(
                    chat_id=user_telegram_id,
                    text=f"❌ <b>Your order #{order_id} has been cancelled by support.</b>\n\n"
                        f"If you have questions, please contact us.",
                    parse_mode="HTML"
                )
            except Exception as e:
                print(f"Could not notify user: {e}")

        await query.edit_message_text(
            f"❌ <b>Order #{order_id} cancelled.</b>",
            parse_mode='HTML'
        )

    elif data.startswith("confirm_reject_link_"):
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        link_id = int(data.split("_")[-1])
        # Mark as rejected
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        cursor.execute("UPDATE account_links SET status = 'rejected', approved_by = ? WHERE id = ?", (user_id, link_id))
        conn.commit()
        conn.close()

        await query.edit_message_text(
            f"❌ <b>Link Request #{link_id} has been rejected.</b>",
            parse_mode="HTML"
        )

        # Notify user of rejection (only here!)
        try:
            conn = sqlite3.connect(db.db_path)
            cursor = conn.cursor()
            cursor.execute("SELECT telegram_user_id, panel_username FROM account_links WHERE id = ?", (link_id,))
            row = cursor.fetchone()
            conn.close()
            if row:
                rejected_user_id, panel_username = row
                try:
                    await context.bot.send_message(
                        chat_id=rejected_user_id,
                        text=(
                            f"❌ <b>Your account link request for <code>{panel_username}</code> was rejected.</b>\n\n"
                            "If you believe this was a mistake, please contact support."
                        ),
                        parse_mode="HTML"
                    )
                except Exception as e:
                    print(f"[WARN] Could not notify user {rejected_user_id}: {e}")
        except Exception as e:
            print(f"[WARN] Error in rejection notification: {e}")

        # Refresh admin links view after 2 seconds
        await asyncio.sleep(2)
        pending_links = db.get_pending_link_requests()
        if not pending_links:
            keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            await query.edit_message_text(
                "🔗 <b>Link Requests</b>\n\nNo pending link requests found.",
                reply_markup=reply_markup,
                parse_mode='HTML'
            )
        else:
            links_text = "🔗 <b>Pending Link Requests</b>\n\n"
            keyboard = []
            for link_request in pending_links:
                l_id, telegram_user_id, panel_username, verification_code, created_at, telegram_username = link_request
                try:
                    created_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
                except:
                    created_date = created_at[:16] if created_at else "Unknown"
                if telegram_username:
                    links_text += f"🆔 <b>#{l_id}</b>\n"
                    links_text += f"👤 <a href='tg://user?id={telegram_user_id}'>@{telegram_username}</a> (<code>{telegram_user_id}</code>)\n"
                else:
                    links_text += f"🆔 <b>#{l_id}</b>\n"
                    links_text += f"👤 <code>{telegram_user_id}</code>\n"
                links_text += f"🔑 Username: <b>{panel_username}</b>\n"
                links_text += f"📝 Code: <code>{verification_code}</code>\n"
                links_text += f"📅 {created_date}\n\n"
                keyboard.append([InlineKeyboardButton(f"✅ Approve #{l_id}", callback_data=f"approve_link_{l_id}")])
                keyboard.append([InlineKeyboardButton(f"❌ Reject #{l_id}", callback_data=f"reject_link_{l_id}")])
            keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
            reply_markup = InlineKeyboardMarkup(keyboard)
            await query.edit_message_text(
                links_text,
                reply_markup=reply_markup,
                parse_mode='HTML',
                disable_web_page_preview=True)
    
    elif data == "admin_prune":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return
        
        await query.edit_message_text("🧹 <b>Pruning Dead Accounts...</b>\n\nThis may take a moment...", parse_mode='HTML')
        
        # Run pruning
        await prune_dead_accounts()
        
        keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            "🧹 <b>Account Pruning Complete</b>\n\nDead and expired accounts have been automatically removed.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data == "admin_stats":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return
        
        # Get system statistics
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        
        # Count users
        cursor.execute("SELECT COUNT(*) FROM users")
        total_users = cursor.fetchone()[0]
        
        # Count linked accounts
        cursor.execute("SELECT COUNT(*) FROM account_links WHERE status = 'approved'")
        total_linked = cursor.fetchone()[0]
        
        # Count orders
        cursor.execute("SELECT COUNT(*) FROM orders")
        total_orders = cursor.fetchone()[0]
        
        cursor.execute("SELECT COUNT(*) FROM orders WHERE status = 'completed'")
        completed_orders = cursor.fetchone()[0]
        
        cursor.execute("SELECT COUNT(*) FROM orders WHERE status = 'pending_payment'")
        pending_orders = cursor.fetchone()[0]
        
        # Get recent activity
        cursor.execute("SELECT COUNT(*) FROM orders WHERE created_at > datetime('now', '-24 hours')")
        orders_24h = cursor.fetchone()[0]
        
        conn.close()
        
        services = package_manager.get_services()
        
        stats_text = f"📊 <b>System Statistics</b>\n\n"
        stats_text += f"👥 <b>Users:</b> {total_users}\n"
        stats_text += f"🔗 <b>Linked Accounts:</b> {total_linked}\n"
        stats_text += f"📦 <b>Active Services:</b> {len(services)}\n\n"
        stats_text += f"📋 <b>Orders:</b>\n"
        stats_text += f"• Total: {total_orders}\n"
        stats_text += f"• Completed: {completed_orders}\n"
        stats_text += f"• Pending: {pending_orders}\n"
        stats_text += f"• Last 24h: {orders_24h}\n\n"
        stats_text += f"🏢 <b>Services:</b>\n"
        
        for service in services:
            service_emoji = "📱"  # Default
            for service_num in range(1, 6):
                if service['id'] == f"service{service_num}":
                    emoji_map = {1: "🎬", 2: "🎭", 3: "📺", 4: "🌐", 5: "📡"}
                    service_emoji = emoji_map.get(service_num, "📱")
                    break
            stats_text += f"• {service_emoji} {service['name']}\n"
        
        keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(stats_text, reply_markup=reply_markup, parse_mode='HTML')

    elif data == "admin_reset_stats":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        keyboard = [
            [InlineKeyboardButton("⚠️ Confirm Reset", callback_data="admin_confirm_reset_stats")],
            [InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "⚠️ <b>Reset ALL Bot Stats?</b>\n\n"
            "This will delete ALL orders and account links. Are you sure?",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )

    elif data == "admin_confirm_reset_stats":
        if not db.is_admin(user_id):
            await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
            return

        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM orders")
        cursor.execute("DELETE FROM account_links")
        conn.commit()
        conn.close()

        await query.edit_message_text(
            "🧨 <b>All stats have been reset.</b>\n\nOrders and account links are now empty.",
            parse_mode='HTML'
        )

    elif data.startswith("select_service_"):
        parts = data.split("_")
        if len(parts) >= 4:
            service_id = parts[-1]
            order_type = "_".join(parts[2:-1])
            await show_billing_cycles(context, chat_id, message_id, order_type, service_id)

    elif data.startswith("select_direct_cycle_"):
        # Special handling for add_funds which has underscores in cycle IDs
        if "add_funds_add_funds_" in data:
            # Extract the cycle ID (everything after the last add_funds_)
            cycle_id_part = data.split("add_funds_add_funds_")[1]  # e.g., "5_dollars"
            cycle_id = cycle_id_part
            service_id = "add_funds"
            order_type = "add_funds"
        else:
            # Normal parsing for other services
            parts = data.split("_")
            if len(parts) >= 6:
                cycle_id = parts[-1]
                service_id = parts[-2]
                order_type = "_".join(parts[3:-2])
            else:
                await query.answer("Invalid selection", show_alert=True)
                return
        
        await show_order_confirmation_direct(context, chat_id, message_id, order_type, service_id, cycle_id)

    elif data.startswith("select_matrix_cycle_"):
        parts = data.split("_")
        if len(parts) >= 6:
            cycle_id = parts[-1]
            service_id = parts[-2]
            order_type = "_".join(parts[3:-2])
            await show_connections(context, chat_id, message_id, order_type, service_id, cycle_id)

    elif data.startswith("select_connection_"):
        parts = data.split("_")
        if len(parts) >= 7:
            connection_count = int(parts[-1])
            cycle_id = parts[-2]
            service_id = parts[-3]
            order_type = "_".join(parts[2:-3])
            await show_xxx_addon_matrix(context, chat_id, message_id, order_type, service_id, cycle_id, connection_count)

    elif data.startswith("xxx_matrix_"):
        parts = data.split("_")
        if len(parts) >= 7:
            xxx_choice = parts[-1]
            connection_count = int(parts[-2])
            cycle_id = parts[-3]
            service_id = parts[-4]
            order_type = "_".join(parts[2:-4])
            xxx_addon = xxx_choice == "yes"
            await show_order_confirmation_matrix(context, chat_id, message_id, order_type, service_id, cycle_id, connection_count, xxx_addon)

    elif data.startswith("confirm_direct_"):
        # Special handling for add_funds which has underscores in cycle IDs
        if "add_funds_add_funds_" in data:
            cycle_id_part = data.split("add_funds_add_funds_")[1]  # e.g., "5_dollars"
            cycle_id = cycle_id_part
            service_id = "add_funds"
            order_type = "add_funds"
        else:
            parts = data.split("_")
            if len(parts) >= 5:
                cycle_id = parts[-1]
                service_id = parts[-2]
                order_type = "_".join(parts[2:-2])
            else:
                await query.answer("Invalid selection", show_alert=True)
                return

        service = package_manager.get_service_by_id(service_id)
        cycle = package_manager.get_billing_cycle(service_id, cycle_id)

        if service and cycle:
            is_wallet_credit = service.get('order_type') == 'wallet_credit'
            if is_wallet_credit:
                username, password = "", ""
            else:
                username, password = generate_username_password()

            base_price = cycle['price']
            setup_fee = 0 if order_type == "renewal" or is_wallet_credit else service.get('setup_fee', 0)
            total_price = base_price + setup_fee

            # Create order with payment tracking ONCE
            order_id, bot_payment_ref = db.add_order_with_payment(
                user_id, query.from_user.username, order_type, service_id, service['name'],
                cycle_id, cycle['name'], cycle['cycle'], cycle.get('panel_package_id', ''),
                base_price, False, setup_fee, total_price, service.get('cashapp_tag', '$Service918'),
                username, password, f"Wallet credit: {cycle['name']}" if is_wallet_credit else f"Direct cycle: {cycle['name']}"
            )

            # Show payment instructions ONCE
            await show_payment_instructions_with_wallet(
                context, chat_id, message_id, order_id, service, cycle, cycle,
                total_price, False, bot_payment_ref, user_id, order_type
            )

    elif data.startswith("confirm_matrix_"):
        parts = data.split("_")
        if len(parts) >= 7:
            xxx_addon = parts[-1] == "True"
            connection_count = int(parts[-2])
            cycle_id = parts[-3]
            service_id = parts[-4]
            order_type = "_".join(parts[2:-4])
            
            service = package_manager.get_service_by_id(service_id)
            cycle = package_manager.get_billing_cycle(service_id, cycle_id)
            connection = package_manager.get_connection_option(service_id, cycle_id, connection_count)
            
            if service and cycle and connection:
                username, password = generate_username_password()
                
                base_price = connection['base_price']
                setup_fee = 0 if order_type == "renewal" else service.get('setup_fee', 0)
                total_price = base_price + setup_fee
                
                # Get the correct panel package ID based on XXX addon
                panel_package_id = connection['panel_package_with_xxx'] if xxx_addon else connection['panel_package_no_xxx']
                
                # Create order with payment tracking
                order_id, bot_payment_ref = db.add_order_with_payment(
                    user_id, query.from_user.username, order_type, service_id, service['name'],
                    f"{cycle_id}_{connection_count}c", f"{cycle['name']} - {connection['name']}", cycle['cycle'], panel_package_id,
                    base_price, xxx_addon, setup_fee, total_price, service.get('cashapp_tag', '$Service918'),
                    username, password, f"Matrix: {cycle['name']}, {connection['name']}, XXX: {'Yes' if xxx_addon else 'No'}"
                )
                
                # Create a pseudo-package object for matrix services
                matrix_package = {
                    'name': f"{cycle['name']} - {connection['name']}",
                    'description': f"{connection['count']} connection(s)"
                }
                
                # ✅ Pass the matrix_package object
                await show_payment_instructions_with_wallet(context, chat_id, message_id, order_id, service, matrix_package, cycle, total_price, xxx_addon, bot_payment_ref, user_id, order_type)

    # Answer the callback to prevent timeout
    await query.answer()

async def check_about_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Debug command to check about.json file"""
    import os
    
    if os.path.exists("about.json"):
        try:
            with open("about.json", "r", encoding="utf-8") as f:
                content = f.read()
            await update.message.reply_text(
                f"✅ about.json exists\n"
                f"Size: {len(content)} bytes\n"
                f"First 200 chars:\n<code>{content[:200]}...</code>",
                parse_mode="HTML"
            )
        except Exception as e:
            await update.message.reply_text(f"❌ Error reading file: {e}")
    else:
        # List files in current directory
        files = os.listdir(".")
        await update.message.reply_text(
            f"❌ about.json not found!\n"
            f"Current directory files:\n{chr(10).join(files[:20])}"
        )

async def show_admin_orders(query, context):
    user_id = query.from_user.id
    if not db.is_admin(user_id):
        await query.edit_message_text("❌ Access Denied", parse_mode='HTML')
        return

    pending_orders = db.get_pending_orders()
    if not pending_orders:
        keyboard = [[InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await query.edit_message_text(
            "📋 <b>Pending Orders</b>\n\nNo pending orders found.",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
    else:
        orders_text = "📋 <b>Pending Orders</b>\n\n"
        keyboard = []
        for order in pending_orders[:10]:  # Show first 10
            order_id, telegram_user_id, order_type, service_name, package_name, billing_cycle, total_price, username, details, created_at, telegram_username = order

            try:
                created_date = datetime.fromisoformat(created_at).strftime("%m/%d %H:%M")
            except:
                created_date = created_at[:16] if created_at else "Unknown"

            orders_text += f"🆔 <b>#{order_id}</b> - ${total_price:.2f}\n"
            orders_text += f"👤 @{telegram_username} ({telegram_user_id})\n"
            orders_text += f"📦 {service_name} - {package_name}\n"
            orders_text += f"📅 {created_date} • {order_type}\n\n"

        keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
        reply_markup = InlineKeyboardMarkup(keyboard)

        await query.edit_message_text(orders_text, reply_markup=reply_markup, parse_mode='HTML')

async def handle_message(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Handle text messages based on user state"""
    user = update.effective_user
    user_id = user.id
    message_text = update.message.text

    if is_admin_only_mode() and user_id not in ADMIN_USER_IDS:
        await update.message.reply_text(
            "🚧 <b>The bot is currently in admin-only mode for maintenance or updates.</b>\n\n"
            "Please try again later.",
            parse_mode='HTML'
        )
        return

    # Check if user has a Telegram username
    if not user.username:
        bot_name = os.getenv("BOT_NAME", "Bot Management System")
        username_required_text = f"""❌ <b>Telegram Username Required</b>

Hello {user.first_name}!

To use this bot, you need to set up a Telegram username (@username).

<b>📝 How to set up your username:</b>

1. Open Telegram Settings
2. Tap on "Username" 
3. Create a unique username (letters, numbers, underscores)
4. Save your changes
5. Return here and type /start again

<b>Why do we need this?</b>
• Account security and verification
• Order tracking and support
• Account linking verification

Please set up your username and try again!"""
        await update.message.reply_text(username_required_text, parse_mode='HTML')
        return

    state, data = db.get_user_state(user_id)

    if state == "link_account_username":
        username = message_text.strip()
        service_name = data
        if len(username) < 3:
            await update.message.reply_text("❌ Username too short. Please enter a valid username.")
            return
        import random
        verification_code = str(random.randint(100000, 999999))
        result = db.request_account_link(user_id, username, verification_code, service_name)
        keyboard = [[InlineKeyboardButton("🏠 Main Menu", callback_data="back_to_menu")]]
        reply_markup = InlineKeyboardMarkup(keyboard)
        if result == "success":
            db.clear_user_state(user_id)
            await update.message.reply_text(
                f"✅ <b>Link Request Submitted</b>\n\n"
                f"Service: <b>{service_name}</b>\n"
                f"Username: <code>{username}</code>\n"
                f"Verification code: <code>{verification_code}</code>\n\n"
                f"Your request has been received and is pending admin approval.\n"
                f"An admin will review your request soon.",
                parse_mode="HTML",
                reply_markup=reply_markup
            )
            return
        elif result == "linked_to_other":
            await update.message.reply_text(
                "❌ <b>Account Already Linked to Another User</b>\n\n"
                "Sorry, this account is already linked to a different Telegram user and cannot be linked again.\n\n"
                "If you believe this is a mistake, please contact support.",
                parse_mode='HTML',
                reply_markup=reply_markup
            )
            return
        elif result == "already_pending":
            await update.message.reply_text(
                "❌ <b>Link Request Already Pending</b>\n\n"
                "You already have a pending link request for this account. Please wait for admin approval or contact support if you need help.",
                parse_mode='HTML',
                reply_markup=reply_markup
            )
            return
        else:
            await update.message.reply_text(
                "❌ <b>Account Already Linked</b>\n\n"
                "This username is already linked to your account or has a pending request.\n\n"
                "Try a different username or contact support if you believe this is an error.",
                parse_mode='HTML',
                reply_markup=reply_markup
            )
            return

    if state == "admin_userlist_search":
        search_term = message_text.strip()
        db.clear_user_state(user_id)
        conn = sqlite3.connect(db.db_path)
        cursor = conn.cursor()
        cursor.execute(
            "SELECT user_id, telegram_username, is_admin, created_at FROM users WHERE telegram_username LIKE ? OR user_id LIKE ? ORDER BY created_at DESC",
            (f"%{search_term}%", f"%{search_term}%")
        )
        all_users = cursor.fetchall()
        conn.close()
        max_per_page = 10
        page = 0
        start = page * max_per_page
        end = start + max_per_page
        page_users = all_users[start:end]
        userlist_text = "👥 <b>User List</b>\n\n"
        userlist_text += f"🔍 <i>Search results for:</i> <b>{search_term}</b>\n\n"
        if not page_users:
            userlist_text += "<i>No users found.</i>\n"
        else:
            for u_id, t_username, is_admin, created_at in page_users:
                label = f"{'👑' if is_admin else ''} @{t_username or '(no_username)'} ({u_id})"
                userlist_text += f"• <a href='tg://user?id={u_id}'>{label}</a>\n"
        keyboard = []
        nav_row = []
        if end < len(all_users):
            nav_row.append(InlineKeyboardButton("➡️ Next", callback_data=f"admin_userlist|search|{search_term}|{page+1}"))
        if nav_row:
            keyboard.append(nav_row)
        keyboard.append([InlineKeyboardButton("🔎 Search Users", callback_data="admin_userlist_search")])
        keyboard.append([InlineKeyboardButton("« Back to Admin", callback_data="admin_panel")])
        reply_markup = InlineKeyboardMarkup(keyboard)
        await update.message.reply_text(
            userlist_text,
            reply_markup=reply_markup,
            parse_mode='HTML',
            disable_web_page_preview=True
        )
        return

    if state == "admin_add_funds_target":
        value = message_text.strip()
        target_id = None
        if value.isdigit():
            target_id = int(value)
        elif value.startswith("@"):
            username = value[1:]
            conn = sqlite3.connect(db.db_path)
            cursor = conn.cursor()
            cursor.execute("SELECT user_id FROM users WHERE telegram_username = ?", (username,))
            row = cursor.fetchone()
            conn.close()
            if row:
                target_id = row[0]
        else:
            username = value
            conn = sqlite3.connect(db.db_path)
            cursor = conn.cursor()
            cursor.execute("SELECT user_id FROM users WHERE telegram_username = ?", (username,))
            row = cursor.fetchone()
            conn.close()
            if row:
                target_id = row[0]
        if not target_id:
            await update.message.reply_text(
                "❌ User not found by that ID or username. Please enter a valid Telegram ID or @username.",
                parse_mode='HTML'
            )
            return
        db.set_user_state(user_id, "admin_add_funds_note", str(target_id))
        await update.message.reply_text(
            f"User found: <code>{target_id}</code>\n\n"
            f"Enter a reason or note for this wallet credit (this will show in their transaction history):",
            parse_mode='HTML'
        )
        return

    if state == "admin_add_funds_note":
        target_id = data
        note = message_text.strip()
        if not note:
            await update.message.reply_text("❌ Please enter a reason or note for the transaction.")
            return
        db.set_user_state(user_id, "admin_add_funds_amount", f"{target_id}|{note}")
        await update.message.reply_text(
            f"Note recorded.\nNow enter the amount to add (USD):",
            parse_mode='HTML'
        )
        return

    if state == "admin_add_funds_amount":
        try:
            target_id, note = data.split("|", 1)
        except Exception:
            await update.message.reply_text("❌ Internal error. Please try again from the admin panel.")
            db.clear_user_state(user_id)
            return
        try:
            amount = float(message_text.strip())
            if amount <= 0:
                raise ValueError()
        except Exception:
            await update.message.reply_text("❌ Enter a valid positive number amount.", parse_mode='HTML')
            return
        db.add_wallet_funds(int(target_id), amount, note)
        db.clear_user_state(user_id)
        await update.message.reply_text(
            f"✅ <b>Added ${amount:.2f} to user {target_id}'s wallet.</b>\nReason: {note}",
            parse_mode='HTML'
        )
        try:
            await context.bot.send_message(
                int(target_id),
                f"💰 <b>Funds Added</b>\n\n"
                f"An admin has credited <b>${amount:.2f}</b> to your wallet.\n"
                f"📝 <b>Note:</b> {note}",
                parse_mode='HTML'
            )
        except Exception:
            pass
        keyboard = [
            [InlineKeyboardButton("🔙 Return to Admin Panel", callback_data="admin_panel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await update.message.reply_text(
            "🔙 <b>Return to Admin Panel</b>",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
        return

    if state == "admin_remove_funds_target":
        value = message_text.strip()
        target_id = None
        if value.isdigit():
            target_id = int(value)
        elif value.startswith("@"):
            username = value[1:]
            conn = sqlite3.connect(db.db_path)
            cursor = conn.cursor()
            cursor.execute("SELECT user_id FROM users WHERE telegram_username = ?", (username,))
            row = cursor.fetchone()
            conn.close()
            if row:
                target_id = row[0]
        else:
            username = value
            conn = sqlite3.connect(db.db_path)
            cursor = conn.cursor()
            cursor.execute("SELECT user_id FROM users WHERE telegram_username = ?", (username,))
            row = cursor.fetchone()
            conn.close()
            if row:
                target_id = row[0]
        if not target_id:
            await update.message.reply_text(
                "❌ User not found by that ID or username. Please enter a valid Telegram ID or @username.",
                parse_mode='HTML'
            )
            return
        db.set_user_state(user_id, "admin_remove_funds_note", str(target_id))
        await update.message.reply_text(
            f"User found: <code>{target_id}</code>\n\n"
            f"Enter a reason or note for this fund removal (this will show in their transaction history):",
            parse_mode='HTML'
        )
        return

    if state == "admin_remove_funds_note":
        target_id = data
        note = message_text.strip()
        if not note:
            await update.message.reply_text("❌ Please enter a reason or note for the transaction.")
            return
        db.set_user_state(user_id, "admin_remove_funds_amount", f"{target_id}|{note}")
        await update.message.reply_text(
            f"Note recorded.\nNow enter the amount to remove (USD):",
            parse_mode='HTML'
        )
        return

    if state == "admin_remove_funds_amount":
        try:
            target_id, note = data.split("|", 1)
        except Exception:
            await update.message.reply_text("❌ Internal error. Please try again from the admin panel.")
            db.clear_user_state(user_id)
            return
        try:
            amount = float(message_text.strip())
            if amount <= 0:
                raise ValueError()
        except Exception:
            await update.message.reply_text("❌ Enter a valid positive number amount.", parse_mode='HTML')
            return
        balance = db.get_wallet_balance(int(target_id))
        if amount > balance:
            await update.message.reply_text(
                f"❌ User only has ${balance:.2f} in their wallet. Cannot remove ${amount:.2f}.\n"
                f"Please enter a smaller amount.",
                parse_mode='HTML'
            )
            return
        db.deduct_wallet_funds(int(target_id), amount, note)
        db.clear_user_state(user_id)
        await update.message.reply_text(
            f"✅ <b>Removed ${amount:.2f} from user {target_id}'s wallet.</b>\nReason: {note}",
            parse_mode='HTML'
        )
        try:
            await context.bot.send_message(
                int(target_id),
                f"💸 <b>Funds Removed</b>\n\n"
                f"An admin has removed <b>${amount:.2f}</b> from your wallet.\n"
                f"📝 <b>Note:</b> {note}",
                parse_mode='HTML'
            )
        except Exception:
            pass
        keyboard = [
            [InlineKeyboardButton("🔙 Return to Admin Panel", callback_data="admin_panel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        await update.message.reply_text(
            "🔙 <b>Return to Admin Panel</b>",
            reply_markup=reply_markup,
            parse_mode='HTML'
        )
        return

    # If no state matched, handle support tickets
    await handle_user_message(update, context)
    await handle_admin_message(update, context)
    return

    # (No fallback else needed, since the ticket handlers should handle the "no state" case)

async def unknown_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Handle unknown commands"""
    await update.message.reply_text(
        "❓ <b>Unknown Command</b>\n\n"
        "I don't recognize that command. Use /start to begin or /help for assistance.",
        parse_mode='HTML'
    )

async def notify_users_of_renewable_accounts(bot_db, bot, days_window=10):
    from datetime import datetime
    import sqlite3

    now = datetime.now().date()
    all_links = bot_db.get_all_linked_accounts()
    for link in all_links:
        telegram_user_id = link[0]
        account_username = link[1]
        expiry_str = link[2]
        service_name = link[3] if len(link) > 3 else "your service"

        if not expiry_str:
            continue
        try:
            expiry_date = datetime.strptime(expiry_str, "%Y-%m-%d").date()
        except Exception:
            continue

        days_left = (expiry_date - now).days
        if 0 < days_left <= days_window:
            # Check if already notified
            conn = sqlite3.connect(bot_db.db_path)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT 1 FROM renewal_notifications
                WHERE telegram_user_id=? AND account_username=? AND expiry_date=?
            """, (telegram_user_id, account_username, expiry_str))
            already_notified = cursor.fetchone()
            if already_notified:
                conn.close()
                continue

            keyboard = InlineKeyboardMarkup([
                [InlineKeyboardButton("🔄 Renew Now", callback_data=f"renew_user_{account_username}")]
            ])

            # Send notification
            try:
                await bot.send_message(
                    telegram_user_id,
                    f"🔄 <b>Renewal Available</b>\n\n"
                    f"Your <b>{service_name}</b> account (<code>{account_username}</code>) "
                    f"is expiring in <b>{days_left} days</b> and is now eligible for renewal.\n"
                    "Renew early to avoid service interruption!",
                    parse_mode="HTML",
                    reply_markup=keyboard
                )
            except Exception as e:
                print(f"Could not notify user {telegram_user_id}: {e}")

            # Mark as notified
            cursor.execute("""
                INSERT INTO renewal_notifications (telegram_user_id, account_username, expiry_date, notified_on)
                VALUES (?, ?, ?, ?)
            """, (telegram_user_id, account_username, expiry_str, now.isoformat()))
            conn.commit()
            conn.close()

async def process_auto_renewals(context: ContextTypes.DEFAULT_TYPE):
    """Process automatic renewals for users with auto-renewal enabled"""
    logging.info("🔄 Processing auto-renewals...")
    
    conn = sqlite3.connect(db.db_path)
    cursor = conn.cursor()
    
    # Get all users with auto-renewal enabled
    cursor.execute("""
        SELECT ar.user_id, ar.payment_timing
        FROM auto_renewal_settings ar
        WHERE ar.enabled = 1
    """)
    auto_renewal_users = cursor.fetchall()
    conn.close()
    
    processed_count = 0
    
    for user_id, timing in auto_renewal_users:
        # Get user's linked accounts
        linked_accounts = db.get_linked_accounts(user_id)
        
        for username, service_name in linked_accounts:
            # Check renewal eligibility
            can_renew, renewal_message = check_renewal_eligibility(username, service_name, panel_manager)
            
            if not can_renew:
                continue
            
            # Parse days until expiry from message
            days_match = re.search(r'expires in (\d+) days', renewal_message)
            if not days_match:
                continue
                
            days_until_expiry = int(days_match.group(1))
            
            # Check timing preference
            should_process = False
            if timing == "immediate" and days_until_expiry <= 10:
                should_process = True
            elif timing == "last_day" and days_until_expiry <= 1:
                should_process = True
            
            if not should_process:
                continue
            
            # Get service and pricing
            service = package_manager.get_service_by_name(service_name)
            if not service or not service.get('billing_cycles'):
                continue
            
            # Use default (first) billing cycle
            default_cycle = service['billing_cycles'][0]
            renewal_price = default_cycle.get('price', 0)
            
            # Check wallet balance
            wallet_balance = db.get_wallet_balance(user_id)
            
            if wallet_balance < renewal_price:
                # Log insufficient funds
                db.log_auto_renewal_attempt(
                    user_id, username, service_name, 'insufficient_funds',
                    renewal_price, wallet_balance, wallet_balance,
                    f"Required: ${renewal_price:.2f}, Available: ${wallet_balance:.2f}"
                )
                
                # Notify user once per day
                notification_key = f"auto_renewal_insufficient_{user_id}_{username}"
                if not has_been_notified_today(notification_key):
                    await context.bot.send_message(
                        user_id,
                        f"💳 <b>Auto-Renewal: Insufficient Funds</b>\n\n"
                        f"Account: <code>{username}</code>\n"
                        f"Service: {service_name}\n"
                        f"Renewal Cost: ${renewal_price:.2f}\n"
                        f"Wallet Balance: ${wallet_balance:.2f}\n\n"
                        f"Please add funds to enable auto-renewal.",
                        parse_mode='HTML'
                    )
                    mark_notified_today(notification_key)
                continue
            
            # Process the renewal
            logging.info(f"AUTO-RENEWAL: Processing {username} for user {user_id}")

            # Get telegram username
            telegram_username = db.get_telegram_username(user_id)

            # Deduct from wallet
            if db.deduct_wallet_funds(user_id, renewal_price, f"Auto-renewal: {username} ({service_name})"):
                # Get the panel package ID for renewal
                panel_package_id = default_cycle.get('panel_package_id', '')
                
                # Create renewal order
                order_id, bot_payment_ref = db.add_order_with_payment(
                    user_id, telegram_username, "renewal",  # Note: using "renewal" type
                    service['id'], service_name, default_cycle['cycle'], 
                    default_cycle['name'], default_cycle['cycle'], 
                    panel_package_id, renewal_price, 
                    False, 0, renewal_price, 'WALLET', username, '',  # username is the existing account
                    f"Auto-renewal from wallet"
                )
                
                # Mark as paid immediately
                db.update_order_status(order_id, "processing", "Auto-renewal from wallet")
                
                # Process renewal in panel
                try:
                    # Get panel for this service
                    service_id = service['id']
                    panel = panel_manager.get_panel_for_service(service_id)
                    
                    if panel:
                        # Process the renewal
                        notes = f"Auto-renewal Order #{order_id} - Wallet payment"
                        
                        renewal_result = panel.renew_user_account(
                            username=username,
                            package_id=panel_package_id,
                            duration_days=30,  # Using default 30 days
                            notes=notes
                        )
                        
                        if renewal_result.get('success'):
                            # Update order as completed
                            db.update_order_status(order_id, "completed", 
                                f"Auto-renewal successful. Extended to: {renewal_result.get('new_expiry', 'N/A')}")
                            
                            # Log success
                            new_balance = db.get_wallet_balance(user_id)
                            db.log_auto_renewal_attempt(
                                user_id, username, service_name, 'success',
                                renewal_price, wallet_balance, new_balance
                            )
                            
                            # Notify user with renewal details
                            await context.bot.send_message(
                                user_id,
                                f"✅ <b>Auto-Renewal Processed</b>\n\n"
                                f"Account: <code>{username}</code>\n"
                                f"Service: {service_name}\n"
                                f"Amount: ${renewal_price:.2f}\n"
                                f"New Balance: ${new_balance:.2f}\n"
                                f"New Expiry: {renewal_result.get('new_expiry', 'N/A')}\n\n"
                                f"Your account has been automatically renewed!",
                                parse_mode='HTML'
                            )
                            
                            processed_count += 1
                        else:
                            # Renewal failed - refund the wallet
                            db.add_wallet_funds(user_id, renewal_price, 
                                f"Refund: Failed auto-renewal for {username}")
                            
                            # Update order status
                            db.update_order_status(order_id, "failed", 
                                f"Panel renewal failed: {renewal_result.get('error', 'Unknown error')}")
                            
                            # Log failure
                            db.log_auto_renewal_attempt(
                                user_id, username, service_name, 'failed',
                                renewal_price, wallet_balance, wallet_balance,
                                renewal_result.get('error', 'Panel renewal failed')
                            )
                            
                            # Notify user
                            await context.bot.send_message(
                                user_id,
                                f"❌ <b>Auto-Renewal Failed</b>\n\n"
                                f"Account: <code>{username}</code>\n"
                                f"Service: {service_name}\n\n"
                                f"The renewal could not be processed. Your wallet has been refunded.\n"
                                f"Error: {renewal_result.get('error', 'Unknown error')}\n\n"
                                f"Please try manual renewal or contact support.",
                                parse_mode='HTML'
                            )
                    else:
                        # No panel configured - refund
                        db.add_wallet_funds(user_id, renewal_price, 
                            f"Refund: No panel for auto-renewal {username}")
                        
                        db.update_order_status(order_id, "failed", "No panel configured")
                        
                        logging.error(f"AUTO-RENEWAL: No panel configured for {service_name}")
                        
                except Exception as e:
                    # Error during renewal - refund
                    db.add_wallet_funds(user_id, renewal_price, 
                        f"Refund: Error during auto-renewal for {username}")
                    
                    db.update_order_status(order_id, "failed", f"Error: {str(e)}")
                    
                    logging.error(f"AUTO-RENEWAL: Error processing {username}: {str(e)}")
                    
                    # Notify user
                    await context.bot.send_message(
                        user_id,
                        f"❌ <b>Auto-Renewal Error</b>\n\n"
                        f"Account: <code>{username}</code>\n"
                        f"An error occurred during renewal. Your wallet has been refunded.\n\n"
                        f"Please try manual renewal or contact support.",
                        parse_mode='HTML'
                    )
            else:
                # Failed to deduct funds
                logging.error(f"AUTO-RENEWAL: Failed to deduct wallet funds for {username}")

# ===== This is the expiry push notification block =====

async def notify_users_of_expiry_stages(bot_db, bot):
    """
    Notifies users of upcoming expiry at 10, 3, 1 days before.
    Uses the renewal_notifications table to avoid spam.
    """
    # Define notification stages (days before expiry)
    NOTIFY_DAYS = [10, 3, 1]

    all_links = bot_db.get_all_linked_accounts()
    today = datetime.now().date()

    for link in all_links:
        telegram_user_id = link[0]
        account_username = link[1]
        expiry_str = link[2]
        service_name = link[3] if len(link) > 3 else "your service"

        if not expiry_str:
            continue
        try:
            expiry_date = datetime.strptime(expiry_str, "%Y-%m-%d").date()
        except Exception:
            continue

        days_left = (expiry_date - today).days
        for n_days in NOTIFY_DAYS:
            if days_left == n_days:
                # Check if already notified for this interval
                conn = sqlite3.connect(bot_db.db_path)
                cursor = conn.cursor()
                cursor.execute("""
                    SELECT 1 FROM renewal_notifications
                    WHERE telegram_user_id=? AND account_username=? AND expiry_date=? AND notified_on=?
                """, (telegram_user_id, account_username, expiry_str, f"EXPIRE_{n_days}"))
                already_notified = cursor.fetchone()
                if already_notified:
                    conn.close()
                    continue

                # Send notification
                keyboard = InlineKeyboardMarkup([
                    [InlineKeyboardButton("🔄 Renew Now", callback_data=f"renew_user_{account_username}")]
                ])
                msg = (
                    f"⏰ <b>Heads up!</b>\n\n"
                    f"Your <b>{service_name}</b> account (<code>{account_username}</code>) "
                    f"expires in <b>{n_days} days</b>.\n"
                    "Renew early to avoid interruption!\n"
                )
                try:
                    await bot.send_message(
                        telegram_user_id,
                        msg,
                        parse_mode="HTML",
                        reply_markup=keyboard
                    )
                except Exception as e:
                    print(f"Could not notify user {telegram_user_id}: {e}")

                # Record notification
                cursor.execute("""
                    INSERT INTO renewal_notifications (telegram_user_id, account_username, expiry_date, notified_on)
                    VALUES (?, ?, ?, ?)
                """, (telegram_user_id, account_username, expiry_str, f"EXPIRE_{n_days}"))
                conn.commit()
                conn.close()
# ===== This is the end of expiry push notification block =====

# Admin test notification command:
async def testnotify_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if update.effective_user.id not in ADMIN_USER_IDS:
        await update.message.reply_text("Admins only!")
        return
    await notify_users_of_renewable_accounts(db, context.bot)
    await update.message.reply_text("Renewal notifications triggered.")

async def myid_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text(
        f"🆔 <b>Your Telegram ID:</b> <code>{update.effective_user.id}</code>",
        parse_mode="HTML"
    )

async def plans_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    services = package_manager.get_services()
    msg = "📦 <b>Available Services & Plans:</b>\n\n"
    keyboard = []

    for svc in services:
        msg += f"<b>{svc['name']}</b>:\n"
        if svc.get('order_type') == 'matrix':
            for cycle in svc.get('billing_cycles', []):
                cycle_name = cycle.get('name', cycle.get('cycle'))
                msg += f"  <u>{cycle_name}:</u>\n"
                for conn in cycle.get('connections', []):
                    if conn.get('enabled', True):
                        conn_name = conn.get('name', f"{conn.get('count', '?')} Connections")
                        price = conn.get('base_price', 0)
                        msg += f"    • {conn_name}: ${price:.2f}\n"
                        # Add button for this plan
                        callback_data = f"select_connection_new_customer_{svc['id']}_{cycle['cycle']}_{conn['count']}"
                        keyboard.append([InlineKeyboardButton(
                            f"{svc['name']} - {cycle_name} - {conn_name} (${price:.2f})",
                            callback_data=callback_data
                        )])
            msg += "\n"
        else:  # direct_cycles
            for cycle in svc.get('billing_cycles', []):
                cycle_name = cycle.get('name', cycle.get('cycle'))
                price = cycle.get('price', 0)
                msg += f"  • {cycle_name}: ${price:.2f}\n"
                callback_data = f"confirm_direct_new_customer_{svc['id']}_{cycle['cycle']}"
                keyboard.append([InlineKeyboardButton(
                    f"{svc['name']} - {cycle_name} (${price:.2f})",
                    callback_data=callback_data
                )])
            msg += "\n"

    # Add a "Back to Start" button
    keyboard.append([InlineKeyboardButton("« Back to Start", callback_data="back_to_start")])

    reply_markup = InlineKeyboardMarkup(keyboard)

    await update.message.reply_text(
        msg + "\nTap a plan below to start your order instantly:",
        parse_mode="HTML",
        reply_markup=reply_markup
    )

async def menu_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text(
        "🏠 <b>Main Menu</b>",
        reply_markup=get_main_menu_keyboard(update.effective_user.id),
        parse_mode="HTML"
    )

async def auto_cancel_unpaid_orders(context):
    """Automatically cancel all orders that have been unpaid for more than 12 hours."""
    cutoff = datetime.now() - timedelta(hours=12)
    conn = sqlite3.connect(db.db_path)
    cursor = conn.cursor()
    cursor.execute("""
        SELECT id, telegram_user_id, created_at
        FROM orders
        WHERE (status = 'pending_payment' OR status = 'pending')
        AND (payment_status = 'pending' OR payment_status IS NULL)
    """)
    to_cancel = []
    for order_id, user_id, created_at in cursor.fetchall():
        try:
            # Parse created_at to datetime
            order_time = (
                datetime.fromisoformat(created_at)
                if "T" in created_at or "-" in created_at
                else datetime.strptime(created_at, "%Y-%m-%d %H:%M:%S")
            )
        except Exception:
            continue
        if order_time < cutoff:
            to_cancel.append((order_id, user_id))
    for order_id, user_id in to_cancel:
        cursor.execute(
            "UPDATE orders SET status='cancelled', notes='Auto-cancelled after 12 hours of non-payment', updated_at=CURRENT_TIMESTAMP WHERE id = ?",
            (order_id,)
        )
        # Optional: Notify the user
        try:
            await context.bot.send_message(
                user_id,
                f"❌ <b>Your order #{order_id} was automatically cancelled</b>\n\n"
                "Reason: Payment was not received within 12 hours.\n"
                "If you still want service, please place a new order.",
                parse_mode="HTML"
            )
        except Exception:
            pass
    conn.commit()
    conn.close()
    if to_cancel:
        print(f"[AutoCancel] Cancelled {len(to_cancel)} unpaid order(s).")

async def canceloldorders_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if update.effective_user.id not in ADMIN_USER_IDS:
        await update.message.reply_text("Admins only!")
        return
    await auto_cancel_unpaid_orders(context)
    await update.message.reply_text("Auto-cancellation complete.")

# ===== This is the promotion push notification (admin command) block =====

async def promo_command(update, context):
    if update.effective_user.id not in ADMIN_USER_IDS:
        await update.message.reply_text("❌ Only admins can set promos.")
        return

    args = context.args
    if not args or args[0] == "help":
        await update.message.reply_text(
            "Usage:\n"
            "/promo <service_id> <cycle_id> [connection_count] <sale_price> <duration_minutes>\n"
            "Or: /promo clear <service_id> <cycle_id> [connection_count]\n"
            "Examples:\n"
            "/promo service1 cycle1 2 8.00 60\n"
            "/promo service2 cycle2 15.00 120\n"
            "/promo clear service1 cycle1 2"
        )
        return

    if args[0] == "clear":
        # Clear promo
        if len(args) < 3:
            await update.message.reply_text("Usage: /promo clear <service_id> <cycle_id> [connection_count]")
            return
        service_id, cycle_id = args[1], args[2]
        connection_count = int(args[3]) if len(args) > 3 else None
        promo_manager.clear_promo(service_id, cycle_id, connection_count)
        await update.message.reply_text("✅ Promo cleared.")
        return

    # Set promo
    if len(args) < 4:
        await update.message.reply_text("Usage: /promo <service_id> <cycle_id> [connection_count] <sale_price> <duration_minutes>")
        return

    service_id, cycle_id = args[0], args[1]
    try:
        # For Service1, expect connection_count
        if service_id == "service1":
            connection_count = int(args[2])
            sale_price = float(args[3])
            duration_minutes = int(args[4])
        else:
            connection_count = None
            sale_price = float(args[2])
            duration_minutes = int(args[3])
    except Exception:
        await update.message.reply_text("Invalid arguments. See /promo help.")
        return

    # Get original price
    if service_id == "service1":
        service = package_manager.get_service_by_id(service_id)
        cycle = package_manager.get_billing_cycle(service_id, cycle_id)
        conn = package_manager.get_connection_option(service_id, cycle_id, connection_count)
        if not (service and cycle and conn):
            await update.message.reply_text("Invalid service/cycle/connection.")
            return
        original_price = conn["base_price"]
    else:
        service = package_manager.get_service_by_id(service_id)
        cycle = package_manager.get_billing_cycle(service_id, cycle_id)
        if not (service and cycle):
            await update.message.reply_text("Invalid service/cycle.")
            return
        original_price = cycle["price"]

    promo_manager.set_promo(service_id, cycle_id, connection_count, sale_price, original_price, duration_minutes)

    # Broadcast to users
    msg = (
        f"🔥 <b>Limited Time Promo!</b>\n\n"
        f"<b>Service:</b> {service['name']}\n"
        f"<b>Billing:</b> {cycle['name']}\n"
    )
    if service_id == "service1":
        msg += f"<b>Connections:</b> {conn['name']}\n"
    msg += (
        f"<b>Price:</b> <s>${original_price:.2f}</s> <b>${sale_price:.2f}</b> 🔥\n"
        f"<i>Hurry! This deal ends in {duration_minutes} minutes.</i>"
    )

    # Send to all users
    conn = sqlite3.connect(db.db_path)
    cursor = conn.cursor()
    cursor.execute("SELECT user_id FROM users")
    all_users = [row[0] for row in cursor.fetchall()]
    conn.close()
    sent_count = 0
    for uid in all_users:
        try:
            await context.bot.send_message(uid, msg, parse_mode="HTML")
            sent_count += 1
        except Exception:
            pass

    await update.message.reply_text(f"✅ Promo set and sent to {sent_count} users.")

def main():
    """Main function to run the bot"""
    backup_database()
    cleanup_old_backups(10)
    check_db_sanity()
    if not BOT_TOKEN:
        print("❌ BOT_TOKEN environment variable is required")
        return

    print(f"🤖 Starting {BOT_NAME}...")
    print(f"📦 Package Manager: {len(package_manager.get_services())} services loaded")
    print(f"🔧 Panel Manager: Initialized")
    print(f"💾 Database: Initialized")
    print(f"👨‍💼 Admins: {ADMIN_USER_IDS}")

    # Force set admin rights
    for admin_id in ADMIN_USER_IDS:
        db.set_admin(admin_id, True)

    # Create application
    application = Application.builder().token(BOT_TOKEN).build()

    # Add handlers
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CallbackQueryHandler(handle_returning_user, pattern="^returning_user$"))
    application.add_handler(CommandHandler("menu", menu_command))
    application.add_handler(CommandHandler("menu", dashboard_command))
    application.add_handler(CommandHandler("myid", myid_command))
    application.add_handler(CommandHandler("canceloldorders", canceloldorders_command))
    application.add_handler(CommandHandler("checkabout", check_about_command))
    application.add_handler(CommandHandler("about", about_handler))
    application.add_handler(CommandHandler("debug", debug_command))
    application.add_handler(CommandHandler("plans", plans_command))
    application.add_handler(CommandHandler("promo", promo_command))
    application.add_handler(CommandHandler("promolist", promo_list_command))
    application.add_handler(CommandHandler("promoclear", promoclear_command))
    application.add_handler(CommandHandler("broadcast", broadcast_command))
    application.add_handler(CommandHandler("testnotify", testnotify_command))
    application.add_handler(CallbackQueryHandler(button_callback))
    application.add_handler(MessageHandler(filters.TEXT & ~filters.COMMAND, handle_message))
    application.add_handler(MessageHandler(filters.COMMAND, unknown_command))

    # ====== BEGIN: THE SCHEDULER FIX ======
    async def auto_renewal_notify(context: ContextTypes.DEFAULT_TYPE):
        await notify_users_of_renewable_accounts(db, context.bot)

    # Schedule the job to run every 24 hours (86400 seconds), starting 1 minute after bot startup:
    application.job_queue.run_repeating(auto_renewal_notify, interval=86400, first=60)
    # ====== END: THE SCHEDULER FIX ========

    # Schedule auto-renewal processing every 4 hours
    application.job_queue.run_repeating(process_auto_renewals, interval=14400, first=300)

    # Automatically cancel unpaid orders every hour
    application.job_queue.run_repeating(auto_cancel_unpaid_orders, interval=3600, first=300)

    # ===== This is the job scheduling for expiry notifications block =====
    # Schedule the multi-stage expiry notification job every 24 hours
    application.job_queue.run_repeating(
        lambda context: notify_users_of_expiry_stages(db, context.bot),
        interval=86400,   # every 24h
        first=60          # start after 1 min
    )
# ===== This is the end of job scheduling for expiry notifications block =====

    # Add error handler
    async def error_handler(update: object, context: ContextTypes.DEFAULT_TYPE) -> None:
        """Log errors and notify user"""
        logger.error("Exception while handling an update:", exc_info=context.error)

        # Try to send error message to user if possible
        if isinstance(update, Update) and update.effective_message:
            try:
                await update.effective_message.reply_text(
                    "❌ <b>Something went wrong</b>\n\n"
                    "An error occurred while processing your request. Please try again or contact support.",
                    parse_mode='HTML'
                )
            except Exception:
                pass  # Ignore errors when sending error messages

    application.add_error_handler(error_handler)

    print("🚀 Bot is starting...")
    print("✅ All handlers registered")
    print("⚡ Bot is running! Press Ctrl+C to stop.")

    # Run the bot
    application.run_polling(allowed_updates=Update.ALL_TYPES)

if __name__ == '__main__':
    import asyncio
    try:
        main()
    except KeyboardInterrupt:
        print("\n🛑 Bot stopped by user")
    except Exception as e:
        print(f"❌ Fatal error: {e}")